#pragma once

#include <jni.h>

#include <type_traits>

#include <util/generic/yexception.h>

namespace NJni {
    namespace NPrivate {
        class TException {
        public:
            explicit TException(const char* type)
                : TException(type, "")
            {
            }

            TException(const char* type, const char* msg)
                : Type(type)
                , Message(msg)
            {
            }

            const char* GetType() const {
                return Type;
            }

            const char* GetMessage() const {
                return Message;
            }

        private:
            const char* const Type;
            const char* const Message;
        };

        class TNullPointerException: TException {
        public:
            TNullPointerException()
                : TException("java/lang/NullPointerException")
            {
            }
        };

        class TIllegalArgumentException: TException {
        public:
            explicit TIllegalArgumentException(const char* msg)
                : TException("java/lang/IllegalArgumentException", msg)
            {
            }
        };

        inline void Throw(JNIEnv* env, const char* clsName, const char* msg) {
            const auto cls = env->FindClass(clsName);
            env->ThrowNew(cls, msg);
        }

        inline void Throw(JNIEnv* env, const char* msg) {
            Throw(env, "java/lang/Exception", msg);
        }

        inline void Throw(JNIEnv* env, jthrowable thr) {
            env->ExceptionClear();
            env->Throw(thr);
        }

        inline void Throw(JNIEnv* env, const TException& e) {
            Throw(env, e.GetType(), e.GetMessage());
        }

        template <class T>
        struct TDefaultReturn {
            static T Get() {
                return T();
            }
        };

        template <>
        struct TDefaultReturn<void> {
            static void Get() {
            }
        };
    }

    inline void ThrowIfError(JNIEnv* env) {
        if (const auto e = env->ExceptionOccurred()) {
            env->ExceptionDescribe();
            throw e;
        }
    }

    inline void Require(bool condition, const char* msg = "") {
        if (!condition) {
            throw NPrivate::TIllegalArgumentException(msg);
        }
    }

    template <class T>
    inline T RequireNotNull(T ptr) {
        if (!ptr) {
            throw NPrivate::TNullPointerException();
        }
        return ptr;
    }

    template <class Func>
    typename std::result_of<Func()>::type Call(JNIEnv* env, Func f) {
        try {
            return f();
        } catch (jthrowable e) {
            NPrivate::Throw(env, e);
        } catch (const NPrivate::TException& e) {
            NPrivate::Throw(env, e);
        } catch (...) {
            NPrivate::Throw(env, CurrentExceptionMessage().c_str());
        }
        return NPrivate::TDefaultReturn<typename std::result_of<Func()>::type>::Get();
    }

    template <class TPtr>
    class TCheckedPtr {
    public:
        TCheckedPtr() = default;
        TCheckedPtr(const TCheckedPtr&) = delete;
        TCheckedPtr(TCheckedPtr&&) = default;
        TCheckedPtr(TPtr&& ref)
            : value_(std::move(ref))
        {
        }
        TCheckedPtr& operator=(const TCheckedPtr&) = delete;
        TCheckedPtr& operator=(TCheckedPtr&&) = default;
        TCheckedPtr& operator=(TPtr&& ref) {
            value_ = std::move(ref);
            return *this;
        }

        auto operator->() const {
            if (!value_) {
                throw NPrivate::TNullPointerException();
            }
            return &*value_;
        }

        auto Get() const {
            return &*value_;
        }

    private:
        TPtr value_;
    };
}
