#include "session_api_request.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

namespace {

    std::string toString(const Json::Value& value) {
        std::string strValue = quasar::jsonToString(value);
        strValue.erase(std::remove(strValue.begin(), strValue.end(), '\n'),
                       strValue.end());
        return strValue;
    }

} // namespace

SessionApiRequest::SessionApiRequest(std::shared_ptr<quasar::HttpClient> client, OnExpiredTokenCallback callback)
    : client_(std::move(client))
    , onExpiredTokenCallback_(std::move(callback))
{
}

SessionApiRequest::~SessionApiRequest() {
}
SessionApiRequest& SessionApiRequest::addHeader(const std::string& name,
                                                const std::string& value) {
    headers_[name] = value;
    return *this;
}

SessionApiRequest& SessionApiRequest::setUrl(const std::string& url) {
    url_ = url;
    return *this;
}

SessionApiRequest& SessionApiRequest::setMethod(const std::string& name) {
    data_["method"] = name;
    return *this;
}

SessionApiRequest& SessionApiRequest::setParams(const Json::Value& params) {
    data_["params"] = params;
    return *this;
}

void SessionApiRequest::executeSync() {
    try {
        const auto response = client_->post("api", url_, toString(data_), headers_);
        resultCode_ = response.responseCode;
        if (response.responseCode == 200) {
            result_ = quasar::parseJson(response.body);
        } else {
            YIO_LOG_WARN("Response " << response.responseCode << " "
                                     << response.responseStatus << ": "
                                     << response.body.substr(0, 500));

            const auto parsedBody = quasar::tryParseJson(response.body);
            result_ = parsedBody.value_or(Json::Value::null);

            if (onExpiredTokenCallback_) {
                const auto data = quasar::tryGetJson(result_, "data", Json::Value::null);
                if (response.responseCode == 403 && quasar::tryGetString(data, "text") == "expired_token") {
                    onExpiredTokenCallback_();
                }
            }
        }
    } catch (const std::runtime_error& error) {
        YIO_LOG_ERROR_EVENT("SessionApiRequest.ExecuteSync.Exception", error.what());
        resultCode_ = 1;
        result_ = Json::Value::null;
    }
}

Json::Value SessionApiRequest::getResult() const {
    return result_;
}

int SessionApiRequest::getErrorCode() const {
    return resultCode_;
}

std::string SessionApiRequest::toCurlCmd() const {
    std::string curl = "-X POST";
    for (auto header : headers_) {
        if (header.first == "Authorization") {
            continue;
        }
        curl += " -H '";
        curl += header.first;
        curl += ": ";
        curl += header.second;
        curl += "'";
    }
    curl += " --data '";
    curl += toString(data_);
    curl += "' --compressed ";
    curl += url_;
    return curl;
}
