#include "call.h"

#include "call_transport.h"
#include "notifier.h"
#include "status.h"

#include <yandex_io/callkit/rtc/media/media_session.h>
#include <yandex_io/callkit/util/loop_thread.h>

#include <memory>

using namespace messenger;

void Call::init(const std::string& chatId, const std::string& userName, const std::string& userAvatar,
                const std::string& userGuid, const std::string& callerDeviceId, const std::string& callerPayload,
                rtc::Direction direction,
                std::weak_ptr<CallsController> callsController,
                std::shared_ptr<rtc::MediaSession> mediaSession,
                std::shared_ptr<LoopThread> workerThread,
                std::shared_ptr<CallTransport> callTransport) {
    mediaSession_ = std::move(mediaSession);
    notifier =
        std::make_shared<Notifier>(std::weak_ptr<Call>(shared_from_this()));
    notifier->attachMediaSessionCallbacks(mediaSession_.get());
    chatId_ = chatId;
    userName_ = userName;
    userAvatar_ = userAvatar;
    userGuid_ = userGuid;
    direction_ = direction;
    callerDeviceId_ = callerDeviceId;
    callerPayload_ = callerPayload;
    callsController_ = std::move(callsController);
    workerThread_ = std::move(workerThread);
    userActionDispatcher_ = std::make_shared<UserActionDispatcher>();
    callTransport_ = std::move(callTransport);
    machine_ = CallStateMachine::create(
        mediaSession_->getCallGuid(), callTransport_, direction_, mediaSession_,
        workerThread_, notifier, Status::NEW, userActionDispatcher_);
}

Call::~Call() = default;

void Call::start() {
    if (direction_ == rtc::Direction::OUTGOING) {
        userActionDispatcher_->onStartOutgoingCall.notifyObservers();
    } else {
        userActionDispatcher_->onStartIncomingCall.notifyObservers();
    }
}

void Call::stop() {
    YIO_LOG_INFO("stop()");
    userActionDispatcher_->onStopCall.notifyObservers();
}

void Call::answer() {
    YIO_LOG_INFO("accept()");
    Y_VERIFY(direction_ == rtc::Direction::INCOMING);
    userActionDispatcher_->onAcceptCall.notifyObservers();
}

void Call::decline() {
    YIO_LOG_INFO("decline()");
    Y_VERIFY(direction_ == rtc::Direction::INCOMING);
    userActionDispatcher_->onDeclineCall.notifyObservers();
}

std::string Call::getChatId() const {
    return chatId_;
}

std::string Call::getUserName() const {
    return userName_;
}

std::string Call::getUserAvatar() const {
    return userAvatar_;
}

std::string Call::getUserGuid() const {
    return userGuid_;
}

std::string Call::getCallGuid() const {
    return mediaSession_ ? mediaSession_->getCallGuid() : "";
}

std::string Call::getCallerDeviceId() const {
    return callerDeviceId_;
}

std::string Call::getCallerPayload() const {
    return callerPayload_;
}

rtc::Direction Call::getDirection() const {
    return direction_;
}

std::weak_ptr<CallsController> Call::getCallsController() const {
    return callsController_;
}

Status Call::getStatus() const {
    return machine_->getCallStatus();
}

rtc::MediaSession* Call::getMediaSession() const {
    return mediaSession_.get();
}

bool Call::setUserName(const std::string& value) {
    if (value != userName_) {
        userName_ = value;
        return true;
    }
    return false;
}

bool Call::setUserAvatar(const std::string& value) {
    if (value != userAvatar_) {
        userAvatar_ = value;
        return true;
    }
    return false;
}
