#include "notifier.h"

#include "call.h"

#include <yandex_io/callkit/rtc/utils/utils.h>

#include <yandex_io/libs/logging/logging.h>

using namespace messenger;

Notifier::Notifier(std::weak_ptr<Call> call)
    : call_(call)
{
}

Notifier::~Notifier() = default;

void Notifier::attachMediaSessionCallbacks(rtc::MediaSession* mediaSession) {
    Y_VERIFY(mediaSession);
    statusChangeSubscription_ =
        mediaSession->subscribeStatus([](rtc::MediaSession::Status status) {
            YIO_LOG_DEBUG("onStatusChange(" << rtc::toString(status) << ")");
        });

    const auto weakRef(weak_from(this));

    failureSubscription_ = mediaSession->subscribeFailure(
        [this, weakRef](const std::string& message) {
            if (const auto ref = weakRef.lock()) {
                YIO_LOG_DEBUG("onFailure(%s)" << message);
                notifyFailure(CallTransport::ErrorCode::UNKNOWN, message);
            }
        });
}

Notifier::OnStartSubscription
Notifier::subscribeOnStart(OnStartListener listener) {
    return onStartListeners_.subscribe(std::move(listener));
}

void Notifier::notifyStart() {
    if (auto call = call_.lock()) {
        onStartListeners_.notifyObservers(call->getChatId());
    }
}

Notifier::OnEndSubscription Notifier::subscribeOnEnd(OnEndListener listener) {
    return onEndListeners_.subscribe(std::move(listener));
}

void Notifier::notifyEnd() {
    if (auto call = call_.lock()) {
        onEndListeners_.notifyObservers(call->getChatId());
    }
}

void Notifier::notifyStatusChange() {
    if (auto call = call_.lock()) {
        onStatusChangeListeners.notifyObservers(call->getChatId());
    }
}

Notifier::OnAcceptSubscription
Notifier::subscribeOnAccept(OnAcceptListener listener) {
    return onAcceptListeners_.subscribe(std::move(listener));
}

void Notifier::notifyAccept() {
    if (auto call = call_.lock()) {
        onAcceptListeners_.notifyObservers(call->getChatId());
    }
}

Notifier::OnDeclineSubscription
Notifier::subscribeOnDecline(OnDeclineListener listener) {
    return onDeclineListeners_.subscribe(std::move(listener));
}

void Notifier::notifyDecline() {
    if (auto call = call_.lock()) {
        onDeclineListeners_.notifyObservers(call->getChatId());
    }
}

Notifier::OnFailureSubscription
Notifier::subscribeOnFailure(OnFailureListener listener) {
    return onFailureListeners_.subscribe(std::move(listener));
}

void Notifier::notifyFailure(CallTransport::ErrorCode code,
                             const std::string& message) {
    if (auto call = call_.lock()) {
        std::string descr = CallTransport::toString(code);
        if (!message.empty()) {
            descr += ": " + message;
        }
        onFailureListeners_.notifyObservers(call->getChatId(), code, descr);
    }
}
