#include "call_live_state.h"

#include "call_ending_state.h"
#include "call_state_machine.h"
#include "user_action_dispatcher.h"

#include <yandex_io/callkit/calls/call/notifier.h>
#include <yandex_io/callkit/calls/call/status.h>

#include <yandex_io/callkit/rtc/utils/utils.h>

#include <yandex_io/libs/logging/logging.h>

#include <chrono>

using namespace messenger;

namespace {
    const auto CONNECT_TIMEOUT = std::chrono::seconds(60);
} // namespace

CallLiveState::CallLiveState() {
}

CallLiveState::~CallLiveState() = default;

void CallLiveState::enter(CallStateMachine* machine) {
    stopCallSubscription_ =
        machine->actions->onStopCall.subscribe(machine, [machine] {
            YIO_LOG_INFO("User stops the call");
            machine->setState(std::make_shared<CallEndingState>(true));
        });
    acceptOrStopCallSubscription_ =
        machine->actions->onAcceptOrStopCall.subscribe(machine, [machine] {
            YIO_LOG_INFO("User accepts or stops the call");
            machine->setState(std::make_shared<CallEndingState>(true));
        });

    auto transportListeners = machine->getTransport()->listeners;
    callEndedSubscription_ =
        transportListeners->onCallEnded.subscribe(machine, [machine] {
            YIO_LOG_INFO("CallEnded message received");
            machine->setState(std::make_shared<CallEndingState>(false));
        });

    const auto thisWeakRef(weak_from(this));
    const auto machineWeakRef(weak_from(machine));

    statusChangeSubscription_ = machine->getMediaSession()->subscribeStatus(
        [this, machine, thisWeakRef,
         machineWeakRef](rtc::MediaSession::Status status) {
            if (const auto thisRef = thisWeakRef.lock()) {
                if (const auto machineRef = machineWeakRef.lock()) {
                    YIO_LOG_INFO("onStatusChange(" << rtc::toString(status) << ")");
                    if (status == rtc::MediaSession::Status::CONNECTED) {
                        stopTimeoutTimer();
                        machine->setCallStatus(Status::CONNECTED);
                    } else {
                        startTimeoutTimer(machine);
                        machine->setCallStatus(Status::CONNECTING);
                    }
                    machine->getNotifier()->notifyStatusChange();
                    if (status == rtc::MediaSession::Status::RECONNECTING) {
                        YIO_LOG_INFO("RECONNECTING");
                    }
                }
            }
        });

    machine->setStartDatetime(std::chrono::system_clock::now());
    machine->setCallStatus(Status::CONNECTING);
    machine->getNotifier()->notifyStart();

    if (machine->getMediaSession()->getStatus() !=
        rtc::MediaSession::Status::CONNECTED) {
        startTimeoutTimer(machine);
    } else {
        machine->setCallStatus(Status::CONNECTED);
        machine->getNotifier()->notifyStatusChange();
    }
}

void CallLiveState::exit() {
    stopCallSubscription_.reset();
    acceptOrStopCallSubscription_.reset();
    callEndedSubscription_.reset();
    statusChangeSubscription_.reset();
    timoutExecutor_.reset();
}

void CallLiveState::startTimeoutTimer(CallStateMachine* machine) {
    timoutExecutor_ = ScopedExecutor::create(machine->getWorkerThread());
    timoutExecutor_.executeDelayed(
        bindWeak(machine,
                 [machine] {
                     YIO_LOG_INFO("Connect timeout exceeded");
                     machine->setState(std::make_shared<CallEndingState>(true));
                 }),
        CONNECT_TIMEOUT);
}

void CallLiveState::stopTimeoutTimer() {
    timoutExecutor_.reset();
}

std::string CallLiveState::toString() const {
    return "CallLiveState";
}
