#include "call_state_machine.h"

#include "call_start_awaiting_state.h"
#include "call_state.h"
#include "initial_call_state.h"
#include "user_action_dispatcher.h"

#include <yandex_io/callkit/calls/call/call_transport.h>

#include <yandex_io/callkit/rtc/media/media_session.h>
#include <yandex_io/callkit/util/loop_thread.h>

using namespace messenger;

void CallStateMachine::init(
    const std::string& callUuid, std::shared_ptr<CallTransport> transport,
    rtc::Direction direction, std::shared_ptr<rtc::MediaSession> mediaSession,
    std::shared_ptr<LoopThread> workerThread,
    std::shared_ptr<Notifier> notifier, Status status,
    std::shared_ptr<UserActionDispatcher> userActionDispatcher) {
    actions = std::move(userActionDispatcher);
    callUuid_ = callUuid;
    direction_ = direction;
    transport_ = std::move(transport);
    mediaSession_ = std::move(mediaSession);
    workerThread_ = std::move(workerThread);
    notifier_ = std::move(notifier);
    callStatus_ = status;
    deviceInfo_ = mediaSession_->getDeviceInfo();
    state_ = std::make_shared<InitialCallState>();
    Y_VERIFY(transport_);
    Y_VERIFY(mediaSession_);
    Y_VERIFY(notifier_);
    setState(std::make_shared<CallStartAwaitingState>());
}

CallStateMachine::~CallStateMachine() = default;

void CallStateMachine::setState(std::shared_ptr<CallState> newState) {
    Y_VERIFY(newState);
    YIO_LOG_INFO("Call state: " << state_->toString() << " -> "
                                << newState->toString());
    {
        // exit/enter can finish call that might cause CSM deleted
        auto me = shared_from_this();
        state_->exit();
        state_ = newState;
        state_->enter(me.get());
    }
}

void CallStateMachine::setStartDatetime(
    std::chrono::system_clock::time_point startDatetime) {
    startDatetime_ = startDatetime;
}

CallTransport* CallStateMachine::getTransport() {
    Y_VERIFY(transport_);
    return transport_.get();
}

rtc::MediaSession* CallStateMachine::getMediaSession() {
    Y_VERIFY(mediaSession_);
    return mediaSession_.get();
}

Notifier* CallStateMachine::getNotifier() {
    Y_VERIFY(transport_);
    return notifier_.get();
}

LoopThread* CallStateMachine::getWorkerThread() {
    Y_VERIFY(workerThread_);
    return workerThread_.get();
}

rtc::Direction CallStateMachine::getDirection() {
    return direction_;
}

Status CallStateMachine::getCallStatus() {
    return callStatus_;
}

std::string CallStateMachine::getCallUuid() {
    return callUuid_;
}

void CallStateMachine::setCallStatus(Status status) {
    YIO_LOG_INFO("Call status: " << toString(callStatus_) << " -> "
                                 << toString(status));
    callStatus_ = status;
}

std::shared_ptr<rtc::DeviceInfo> CallStateMachine::createDeviceInfo() {
    return std::make_shared<rtc::DeviceInfo>(*deviceInfo_);
}
