#include "incoming_call_accepting_state.h"

#include "call_ending_state.h"
#include "call_live_state.h"
#include "call_state_machine.h"

#include <yandex_io/callkit/calls/call/notifier.h>
#include <yandex_io/callkit/calls/call/status.h>

using namespace messenger;

namespace {
    const auto ACCEPT_TIMEOUT = std::chrono::seconds(10);
} // namespace

IncomingCallAcceptingState::IncomingCallAcceptingState() {
}

IncomingCallAcceptingState::~IncomingCallAcceptingState() = default;

void IncomingCallAcceptingState::enter(CallStateMachine* machine) {
    auto* transport = machine->getTransport()->listeners.get();
    ackReceivedSubscription_ = transport->onAckReceived.subscribe(
        machine, this, [machine, this](RequestId requestId) {
            if (requestId_ == requestId) {
                YIO_LOG_INFO("AcceptCall ack received for request "
                             << requestId.getId() << ")");
                machine->setState(std::make_shared<CallLiveState>());
            }
        });
    errorReceivedSubscription_ = transport->onErrorReceived.subscribe(
        machine, this,
        [machine, this](RequestId requestId,
                        CallTransport::ErrorCode errorCode) {
            if (requestId_ != requestId) {
                return;
            }
            YIO_LOG_INFO("AcceptCall request failed, " << requestId.getId() << ")");
            machine->setState(std::make_shared<CallEndingState>(false));
        });

    // Notifier attaches listeners to MediaSession by itself.

    timoutExecutor_ = ScopedExecutor::create(machine->getWorkerThread());
    timoutExecutor_.executeDelayed(
        bindWeak(this, machine,
                 [machine] {
                     YIO_LOG_INFO("AcceptCall timeout exceeded");
                     machine->setState(std::make_shared<CallEndingState>(true));
                 }),
        ACCEPT_TIMEOUT);

    machine->setCallStatus(Status::ACCEPTING);

    machine->getNotifier()->notifyStatusChange();

    acceptCall(machine);
}

void IncomingCallAcceptingState::acceptCall(CallStateMachine* machine) {
    requestId_ =
        machine->getTransport()->acceptCall(machine->createDeviceInfo());
}

void IncomingCallAcceptingState::exit() {
    ackReceivedSubscription_.reset();
    errorReceivedSubscription_.reset();
    timoutExecutor_.reset();
}

std::string IncomingCallAcceptingState::toString() const {
    return "IncomingCallAcceptingState";
}
