#include "incoming_call_ringing_state.h"

#include "call_ending_state.h"
#include "call_state_machine.h"
#include "incoming_call_accepting_state.h"
#include "incoming_call_declining_state.h"
#include "user_action_dispatcher.h"

#include <yandex_io/callkit/calls/call/notifier.h>

#include <yandex_io/libs/logging/logging.h>

#include <chrono>

using namespace messenger;

namespace {
    const auto RINGING_TIMEOUT = std::chrono::seconds(60);
} // namespace

IncomingCallRingingState::IncomingCallRingingState() {
}

IncomingCallRingingState::~IncomingCallRingingState() = default;

void IncomingCallRingingState::enter(CallStateMachine* machine) {
    auto* actions = machine->actions.get();
    acceptCallSubscription_ =
        actions->onAcceptCall.subscribe(machine, [machine] {
            machine->setState(std::make_shared<IncomingCallAcceptingState>());
        });
    declineCallSubscription_ =
        actions->onDeclineCall.subscribe(machine, [machine] {
            machine->setState(std::make_shared<IncomingCallDecliningState>());
        });
    stopCallSubscription_ = actions->onStopCall.subscribe(machine, [machine] {
        machine->setState(std::make_shared<IncomingCallDecliningState>());
    });
    acceptOrStopCallSubscription_ =
        actions->onAcceptOrStopCall.subscribe(machine, [machine] {
            machine->setState(std::make_shared<IncomingCallAcceptingState>());
        });

    auto* transport = machine->getTransport()->listeners.get();
    acceptedByOtherSubscription_ =
        transport->onCallAcceptedByOtherClient.subscribe(machine, [machine] {
            YIO_LOG_INFO("Call(callUuid=" << machine->getCallUuid()
                                          << ") accepted by other device");
            machine->setState(std::make_shared<CallEndingState>(false));
        });
    endedSubscription_ = transport->onCallEnded.subscribe(machine, [machine] {
        YIO_LOG_INFO("Call(callUuid=" << machine->getCallUuid()
                                      << ") ended by remote");
        machine->setState(std::make_shared<CallEndingState>(false));
    });
    declinedSubscription_ =
        transport->onCallDeclined.subscribe(machine, [machine] {
            YIO_LOG_INFO("Call(callUuid=" << machine->getCallUuid()
                                          << ") declined by other device");
            machine->setState(std::make_shared<CallEndingState>(false));
        });
    ackReceivedSubscription_ = transport->onAckReceived.subscribe(
        machine, this, [this](RequestId requestId) {
            if (requestId_ == requestId) {
                YIO_LOG_INFO("NotifyRinging ack received");
            }
        });
    errorReceivedSubscription_ = transport->onErrorReceived.subscribe(
        machine, this,
        [machine, this](RequestId requestId,
                        CallTransport::ErrorCode errorCode) {
            if (requestId_ != requestId) {
                return;
            }
            YIO_LOG_INFO("NotifyRinging failed with code="
                         << CallTransport::toString(errorCode));
            if (errorCode == CallTransport::ErrorCode::CONFLICT) {
                // Call has accepted from another client or ended.
                machine->setState(std::make_shared<CallEndingState>(false));
            }
        });

    timoutExecutor_ = ScopedExecutor::create(machine->getWorkerThread());
    timoutExecutor_.executeDelayed(
        bindWeak(machine,
                 [machine] {
                     YIO_LOG_INFO("Connect timeout exceeded");
                     machine->setState(std::make_shared<CallEndingState>(true));
                 }),
        RINGING_TIMEOUT);

    machine->setCallStatus(Status::RINGING);
    machine->getNotifier()->notifyStatusChange();

    requestId_ = machine->getTransport()->notifyRinging();
}

void IncomingCallRingingState::exit() {
    acceptCallSubscription_.reset();
    declineCallSubscription_.reset();
    stopCallSubscription_.reset();
    acceptOrStopCallSubscription_.reset();

    acceptedByOtherSubscription_.reset();
    endedSubscription_.reset();
    declinedSubscription_.reset();
    ackReceivedSubscription_.reset();
    errorReceivedSubscription_.reset();

    timoutExecutor_.reset();
}

std::string IncomingCallRingingState::toString() const {
    return "IncomingCallRingingState";
}
