#include "outgoing_call_accept_awaiting_state.h"

#include "call_ending_state.h"
#include "call_live_state.h"
#include "call_state_machine.h"
#include "user_action_dispatcher.h"

#include <yandex_io/callkit/calls/call/call_transport.h>
#include <yandex_io/callkit/calls/call/notifier.h>
#include <yandex_io/callkit/calls/call/status.h>

#include <yandex_io/callkit/rtc/media/media_session.h>

#include <yandex_io/libs/logging/logging.h>

#include <chrono>

using namespace messenger;

namespace {
    const std::chrono::seconds DIALING_TIMEOUT(60);
} // namespace

OutgoingCallAcceptAwaitingState::OutgoingCallAcceptAwaitingState() = default;

OutgoingCallAcceptAwaitingState::~OutgoingCallAcceptAwaitingState() = default;

void OutgoingCallAcceptAwaitingState::enter(CallStateMachine* machine) {
    onStopCallSubscription_ =
        machine->actions->onStopCall.subscribe(machine, [machine] {
            YIO_LOG_INFO("Cancelled");
            machine->setState(std::make_shared<CallEndingState>(true));
        });

    auto listeners = machine->getTransport()->listeners;
    callAcceptedSubscription_ =
        listeners->onCallAccepted.subscribe(machine, [machine]() {
            YIO_LOG_INFO("Call accepted");
            machine->getNotifier()->notifyAccept();
            machine->setState(std::make_shared<CallLiveState>());
        });
    callOtherAcceptedSubscription_ =
        listeners->onCallAcceptedByOtherClient.subscribe(machine, [machine]() {
            YIO_LOG_INFO("Call accepted");
            machine->getNotifier()->notifyAccept();
            machine->setState(std::make_shared<CallLiveState>());
        });
    callDeclinedSubscription_ =
        listeners->onCallDeclined.subscribe(machine, [machine]() {
            YIO_LOG_INFO("Call declined");
            machine->getNotifier()->notifyDecline();
            machine->setState(std::make_shared<CallEndingState>(false));
        });
    ringingSubscription_ = listeners->onRinging.subscribe(machine, [machine]() {
        YIO_LOG_INFO("Call ringing");
        machine->setCallStatus(Status::RINGING);
        machine->getNotifier()->notifyStatusChange();
    });

    timeoutCallbackRunner_ = ScopedExecutor::create(machine->getWorkerThread());
    timeoutCallbackRunner_.executeDelayed(
        bindWeak(machine,
                 [machine] {
                     YIO_LOG_INFO("Dialing timeout exceeded");
                     machine->getNotifier()->notifyFailure(
                         CallTransport::ErrorCode::TIMEOUT,
                         "CallAcceptTimeout");
                     machine->setState(std::make_shared<CallEndingState>(true));
                 }),
        DIALING_TIMEOUT);
    machine->setCallStatus(Status::DIALING);
    machine->getNotifier()->notifyStatusChange();
    YIO_LOG_INFO("Dialing...");
}

void OutgoingCallAcceptAwaitingState::exit() {
    onStopCallSubscription_.reset();
    callAcceptedSubscription_.reset();
    callOtherAcceptedSubscription_.reset();
    callDeclinedSubscription_.reset();
    ringingSubscription_.reset();
    timeoutCallbackRunner_.reset();
}

std::string OutgoingCallAcceptAwaitingState::toString() const {
    return "OutgoingCallAcceptAwaitingState";
}
