#include "outgoing_call_creating_state.h"

#include "call_ending_state.h"
#include "call_state_machine.h"
#include "outgoing_call_accept_awaiting_state.h"
#include "outgoing_call_creation_failed_state.h"
#include "user_action_dispatcher.h"

#include <yandex_io/callkit/calls/call/notifier.h>

#include <yandex_io/libs/logging/logging.h>

#include <chrono>

using namespace messenger;

namespace {
    const auto MAKE_CALL_TIMEOUT = std::chrono::seconds(10);
} // namespace

OutgoingCallCreatingState::OutgoingCallCreatingState()
    : requestId_(RequestId::null())
{
}

OutgoingCallCreatingState::~OutgoingCallCreatingState() = default;

void OutgoingCallCreatingState::enter(CallStateMachine* machine) {
    onStopCallSubscription_ =
        machine->actions->onStopCall.subscribe(machine, [machine] {
            machine->setState(std::make_shared<CallEndingState>(true));
        });

    auto listeners = machine->getTransport()->listeners;
    auto myRequestId = requestId_;
    ackSubscription_ = listeners->onAckReceived.subscribe(
        this, machine, [this, machine](RequestId requestId) {
            if (requestId_ != requestId) {
                return;
            }
            YIO_LOG_INFO("Call(callUuid=" << machine->getCallUuid()
                                          << " successfully created");
            machine->setState(
                std::make_shared<OutgoingCallAcceptAwaitingState>());
        });
    errorSubscription_ = listeners->onErrorReceived.subscribe(
        this, machine,
        [this, machine](RequestId requestId, CallTransport::ErrorCode err) {
            if (requestId_ != requestId) {
                return;
            }
            YIO_LOG_INFO("Call(callUuid=" << machine->getCallUuid()
                                          << " creation failed (error="
                                          << CallTransport::toString(err) << ")");
            machine->setState(
                std::make_shared<OutgoingCallCreationFailedState>(err));
        });

    // Notifier attaches listeners to MediaSession by itself.

    timoutExecutor_ = ScopedExecutor::create(machine->getWorkerThread());
    timoutExecutor_.executeDelayed(
        bindWeak(this, machine,
                 [machine] {
                     YIO_LOG_INFO("MakeCall timeout exceeded");
                     machine->setState(
                         std::make_shared<OutgoingCallCreationFailedState>(
                             CallTransport::ErrorCode::TIMEOUT));
                 }),
        MAKE_CALL_TIMEOUT);

    makeCall(machine);
}

void OutgoingCallCreatingState::exit() {
    onStopCallSubscription_.reset();
    ackSubscription_.reset();
    errorSubscription_.reset();
    timoutExecutor_.reset();
}

void OutgoingCallCreatingState::makeCall(CallStateMachine* machine) {
    requestId_ = machine->getTransport()->makeCall(machine->createDeviceInfo());
}

std::string OutgoingCallCreatingState::toString() const {
    return "OutgoingCallCreatingState";
}
