#include "calling_message_receiver.h"

#include <mssngr/router/lib/protos/client.pb.h>

#include <yandex_io/libs/logging/logging.h>

#include <chrono>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

CallingMessageReceiver::CallingMessageReceiver(
    std::shared_ptr<CallTransport::Listeners> listeners,
    std::shared_ptr<LoopThread> workerThread, const std::string& deviceId,
    uint64_t receiverSequenceNumber)
    : callTransportListeners_(std::move(listeners))
    , workerThread_(std::move(workerThread))
    , deviceId_(deviceId)
    , expectedSequenceNumber_(receiverSequenceNumber)
{
    Y_VERIFY(workerThread_->checkInside());
}

CallingMessageReceiver::~CallingMessageReceiver() = default;

void CallingMessageReceiver::dispose() {
    YIO_LOG_DEBUG("Disposing receiver: " << this);
    stopTimer();
}

void CallingMessageReceiver::setCallGuid(const std::string& callGuid) {
    Y_VERIFY(workerThread_->checkInside());
    callGuid_ = callGuid;
}

void CallingMessageReceiver::setMediaTransportListener(
    std::weak_ptr<rtc::Transport::Listener> listener) {
    Y_VERIFY(workerThread_->checkInside());
    mediaTransportListener_ = std::move(listener);
}

void CallingMessageReceiver::onCallingMessage(proto::CallingMessage message) {
    Y_VERIFY(workerThread_->checkInside());

    auto seqno = message->seqno();
    if (seqno == 0u) {
        handleCallingMessage(message);
        return;
    }
    if (seqno < expectedSequenceNumber_) {
        YIO_LOG_ERROR_EVENT("CallingMessageReceiver.DuplicateMessage", "Duplicate message detected, sequenceNumber=" << seqno);
        handleCallingMessage(message);
        return;
    }
    queue_[message->seqno()] = std::move(message);
    handleMessageQueue();
}

void CallingMessageReceiver::handleCallingMessage(
    proto::CallingMessage message) {
#ifndef _NDEBUG
    YIO_LOG_DEBUG(message->ShortDebugString());
#endif
    if (message->has_transportmessage()) {
        handleTransportMessage(message->transportmessage().payload());
    } else if (message->has_callaccepted()) {
        handleCallAcceptedMessage(message->callaccepted().accepteddeviceid());
    } else if (message->has_calldeclined()) {
        callTransportListeners_->onCallDeclined.notifyObservers();
    } else if (message->has_callended()) {
        callTransportListeners_->onCallEnded.notifyObservers();
    } else if (message->has_ringing()) {
        callTransportListeners_->onRinging.notifyObservers();
    } else {
        YIO_LOG_WARN("Unexpected CallingMessage received");
    }
}

void CallingMessageReceiver::handleMessageQueue() {
    YIO_LOG_DEBUG("Queue size: " << queue_.size());
    do {
        auto it = queue_.find(expectedSequenceNumber_);
        if (it == queue_.end()) {
            break;
        }
        auto message = std::move(it->second);
        queue_.erase(it);
        stopTimer();
        expectedSequenceNumber_++;
        handleCallingMessage(message);
    } while (true);
    if (!queue_.empty()) {
        YIO_LOG_ERROR_EVENT("CallingMessageReceiver.UnexpectedMessage", "Unexpected message arrived, expected sequenceNumber="
                                                                            << expectedSequenceNumber_);
        startTimer();
    }
}

void CallingMessageReceiver::startTimer() {
    timerExecutor_ = ScopedExecutor::create(workerThread_);
    timerExecutor_.executeDelayed([this] { onTimeout(); },
                                  std::chrono::seconds(1));
}

void CallingMessageReceiver::stopTimer() {
    timerExecutor_.reset();
}

void CallingMessageReceiver::onTimeout() {
    YIO_LOG_ERROR_EVENT("CallingMessageReceiver.Timeout", "Message with sequenceNumber=" << expectedSequenceNumber_
                                                                                         << " is lost.");
    // Skip message with an expected sequence number and try to handle the rest
    // of the queue.
    expectedSequenceNumber_++;
    handleMessageQueue();
}

void CallingMessageReceiver::handleTransportMessage(
    const std::string& transportMessagePayload) {
#ifndef _NDEBUG
    YIO_LOG_DEBUG("TransportMessage: " << transportMessagePayload);
#endif
    if (auto listener = mediaTransportListener_.lock()) {
        listener->onMessage(transportMessagePayload);
    } else {
        YIO_LOG_ERROR_EVENT("CallingMessageReceiver.NullTransportListener",
                            "Can't pass TransportMessage, mediaTransportListener is null");
        return;
    }
}

void CallingMessageReceiver::handleCallAcceptedMessage(
    const std::string& acceptedDeviceId) {
    if (deviceId_ == acceptedDeviceId) {
        callTransportListeners_->onCallAccepted.notifyObservers();
    } else {
        callTransportListeners_->onCallAcceptedByOtherClient.notifyObservers();
    }
}
