#include "calling_messages_sender.h"

#include <mssngr/router/lib/protos/client.pb.h>
#include <yandex_io/callkit/connection/client_request_task.h>
#include <yandex_io/callkit/connection/push_handler.h>
#include <yandex_io/callkit/connection/request_handler.h>
#include <yandex_io/callkit/calls/types.h>
#include <yandex_io/callkit/util/loop_thread.h>

#include <functional>

using namespace messenger;

namespace {

    class CallingMessageRequest: public ClientRequestTask {
    public:
        CallingMessageRequest(const std::string& payloadId,
                              proto::CallingMessage message,
                              std::weak_ptr<CallingMessagesSender> senderWeakRef)
            : payloadId_(payloadId)
            , message_(std::move(message))
            , senderWeakRef_(senderWeakRef)
        {
        }

        std::string getName() override {
            return "CallingMessageRequest";
        }

        void prepareClientMessage(
            NMessengerProtocol::TClientMessage* clientMessage) override {
            auto* callingMessage = clientMessage->mutable_callingmessage();
            *callingMessage = *message_;
        }
        void handleResponse(proto::PostMessageResponse /*message*/) override {
            if (auto sender = senderWeakRef_.lock()) {
                sender->onAckReceived.notifyObservers(payloadId_);
            }
        }
        bool handleError(proto::PostMessageResponse response) override {
            if (auto sender = senderWeakRef_.lock()) {
                sender->onErrorReceived.notifyObservers(payloadId_, response);
            }
            return false;
        }

    private:
        std::string payloadId_;
        proto::CallingMessage message_;
        std::weak_ptr<CallingMessagesSender> senderWeakRef_;
    };

} // namespace

void CallingMessagesSender::init(std::shared_ptr<LoopThread> workerThread,
                                 std::shared_ptr<RequestHandler> requestHandler,
                                 std::shared_ptr<PushHandler> pushHandler) {
    workerThread_ = std::move(workerThread);
    requestHandler_ = std::move(requestHandler);
    pushHandler_ = std::move(pushHandler);
    Y_VERIFY(workerThread_->checkInside());

    pushSubscription_ = pushHandler_->onServerMessage.subscribe(
        shared_from_this(), std::bind(&CallingMessagesSender::onServerMessage,
                                      this, std::placeholders::_1));
}

Cancelable
CallingMessagesSender::sendCallingMessage(const std::string& payloadId,
                                          proto::CallingMessage message) {
    Y_VERIFY(workerThread_->checkInside());

    auto task = std::make_shared<CallingMessageRequest>(
        payloadId, std::move(message),
        std::weak_ptr<CallingMessagesSender>(shared_from_this()));
    return requestHandler_->process(task);
}

void CallingMessagesSender::onServerMessage(proto::ServerMessage message) {
    Y_VERIFY(workerThread_->checkInside());

    if (!message->has_clientmessage()) {
        return;
    }
    const auto& clientMessage = message->clientmessage();
    if (!clientMessage.has_callingmessage()) {
        return;
    }
    YIO_LOG_DEBUG("Calling message received");
    onCallingMessage.notifyObservers(
        proto::clone(clientMessage.callingmessage()), std::move(message));
}
