#include "calling_service.h"

#include "calling_messages_sender.h"
#include "calls_controller.h"
#include "calls_controller_provider.h"
#include "call/call.h"

#include <yandex_io/callkit/connection/push_handler.h>
#include <yandex_io/callkit/connection/request_handler.h>
#include <yandex_io/callkit/session/session_settings.h>
#include <yandex_io/callkit/storage/data_storage.h>
#include <yandex_io/callkit/util/loop_thread.h>

#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>

using namespace messenger;

void CallingService::init(
    const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory,
    const SessionSettings& settings,
    std::shared_ptr<LoopThread> workerThread,
    std::shared_ptr<DataStorage> dataStorage,
    std::shared_ptr<SessionRequestFactory> apiRequestFactory,
    std::shared_ptr<AsyncService> apiExecutor,
    std::shared_ptr<RequestHandler> requestHandler,
    std::shared_ptr<PushHandler> pushHandler,
    Json::Value callsConfig) {
    workerThread_ = std::move(workerThread);
    dataStorage_ = std::move(dataStorage);
    requestHandler_ = std::move(requestHandler);
    pushHandler_ = std::move(pushHandler);
    callHolder_ = std::make_shared<CallHolder>(workerThread_);
    callSubscription_ = callHolder_->onCallChanged.subscribe(
        shared_from_this(),
        [this](std::shared_ptr<Call> call) { processIdleState(); });
    callingMessagesSender_ = CallingMessagesSender::create(
        workerThread_, requestHandler_, pushHandler_);
    callsControllerProvider_ = CallsControllerProvider::create(
        ipcFactory, settings, workerThread_, dataStorage_, apiRequestFactory, apiExecutor,
        requestHandler_, pushHandler_, callingMessagesSender_, callHolder_,
        std::make_shared<quasar::UserConfigProvider>(ipcFactory), std::move(callsConfig));
}

CallingService::~CallingService() = default;

void CallingService::setAllowUsualCalls(bool allowUsualCalls) {
    callsControllerProvider_->setAllowUsualCalls(allowUsualCalls);
}

std::shared_ptr<CallHolder> CallingService::getCallHolder() {
    return callHolder_;
}

void CallingService::startCall(const std::string& userGuid, const std::string& callPayload) {
    auto callsController = callsControllerProvider_->getByUserGuid(userGuid);
    callsController->startOutgoingCall(callPayload);
}

void CallingService::startCallToOwnDevice(const std::string& deviceId, const std::string& callPayload) {
    auto callsController = callsControllerProvider_->getByDeviceId(deviceId);
    callsController->startOutgoingCall(callPayload);
}

void CallingService::declineIncomingCall() {
    auto call = callHolder_->getCall();
    if (!call) {
        return;
    }
    if (auto callsController = call->getCallsController().lock()) {
        callsController->declineIncomingCall();
    }
}

void CallingService::acceptIncomingCall() {
    auto call = callHolder_->getCall();
    if (!call) {
        return;
    }
    if (auto callsController = call->getCallsController().lock()) {
        callsController->acceptIncomingCall();
    }
}

void CallingService::hangupCall() {
    auto call = callHolder_->getCall();
    if (!call) {
        return;
    }

    if (auto callsController = call->getCallsController().lock()) {
        callsController->hangupCall();
    }
}

bool CallingService::hasBgTasks() const {
    return callHolder_->getCall() != nullptr;
}

void CallingService::runOnBgIdle(std::function<void()> callback) {
    idleCallbacks_.push(std::move(callback));
}

void CallingService::processIdleState() {
    if (!hasBgTasks()) {
        while (!idleCallbacks_.empty()) {
            auto callback = idleCallbacks_.front();
            idleCallbacks_.pop();
            if (callback) {
                callback();
            }
        }
    }
}
