#include "calls_controller.h"

#include "calling_messages_sender.h"
#include "messenger_call_transport.h"
#include "call/call.h"
#include "call/call_holder.h"
#include "call/call_transport.h"

#include <mssngr/router/lib/protos/client.pb.h>
#include <yandex_io/callkit/api/create_private_chat_task.h>
#include <yandex_io/callkit/api/session_request_factory.h>
#include <yandex_io/callkit/rtc/media/device_info.h>
#include <yandex_io/callkit/rtc/media/direction.h>
#include <yandex_io/callkit/rtc/media/media_session.h>
#include <yandex_io/callkit/rtc/media/media_session_factory.h>
#include <yandex_io/callkit/session/session_settings.h>
#include <yandex_io/callkit/util/async_service.h>

#include <functional>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

namespace {

    const size_t MAX_CALL_GUID_COUNT = 10;

    class PrepareChatApiTask: public CreatePrivateChatTask {
    public:
        PrepareChatApiTask(
            std::shared_ptr<SessionRequestFactory> apiRequestFactory,
            const std::string& userGuid,
            std::function<void(const Json::Value&, const Json::Value&)> onSuccess,
            std::function<void(int)> onFailed)
            : CreatePrivateChatTask(apiRequestFactory, userGuid)
            , onSuccess_(onSuccess)
            , onFailed_(onFailed)
        {
        }
        std::string getName() override {
            return "PrepareChatApiTask";
        }
        void onError(int code) override {
            if (onFailed_) {
                onFailed_(code);
            }
        }

    protected:
        void onResponse(const Json::Value& userData,
                        const Json::Value& chatData) override {
            if (onSuccess_) {
                onSuccess_(chatData, userData);
            }
        }

    private:
        std::function<void(const Json::Value&, const Json::Value&)> onSuccess_;
        std::function<void(int)> onFailed_;
    };

    std::shared_ptr<rtc::DeviceInfo> getDeviceInfo(const PlatformInfo& info) {
        auto deviceInfo = std::make_shared<rtc::DeviceInfo>();
        Y_VERIFY(!info.appId.empty());
        Y_VERIFY(!info.appName.empty());
        deviceInfo->setAppId(info.appId)
            ->setAppName(info.appName)
            ->setAppVersion(info.appVersion)
            ->setOsVersion(info.osVersion)
            ->setDeviceManufacturer(info.deviceManufacturer)
            ->setDeviceType(info.deviceType)
            ->setDeviceId(info.deviceId)
            ->setDeviceOs(info.deviceOs)
            ->setSoftwareVersion(info.softwareVersion);
        return deviceInfo;
    }

    std::optional<std::string> getDisplayName(const proto::ServerMessage& message) {
        if (message->has_servermessageinfo() && message->servermessageinfo().has_from()) {
            return message->servermessageinfo().from().displayname();
        }

        YIO_LOG_ERROR_EVENT("CallsController.EmptyCallerName", "No caller name in server message");

        return {};
    }

    std::optional<std::string> getAvatarUrl(const proto::ServerMessage& message) {
        std::string id;
        if (message->has_servermessageinfo() && message->servermessageinfo().has_from()) {
            id = message->servermessageinfo().from().avatarid();
        }
        // Note that there are a few other avatar storages, we are aware only of
        // yapic url transformation.
        if (!id.empty() && id.substr(0, 18) == "user_avatar/yapic/") {
            return "https://avatars.mds.yandex.net/get-yapic/" + id.substr(18) + "/islands-300";
        }
        return {};
    }

} // namespace

CallsController::CallsController(
    std::shared_ptr<LoopThread> workerThread,
    std::shared_ptr<SessionRequestFactory> apiRequestFactory,
    std::shared_ptr<AsyncService> apiExecutor,
    std::shared_ptr<PushHandler> pushHandler,
    std::shared_ptr<CallingMessagesSender> callingMessagesSender,
    std::shared_ptr<rtc::MediaSessionFactory> mediaSessionFactory,
    std::shared_ptr<CallHolder> callHolder,
    std::shared_ptr<PlatformInfoProvider> platformInfo,
    const std::string& userGuid,
    const std::string& receiverDeviceId)
    : workerThread_(std::move(workerThread))
    , apiRequestFactory_(std::move(apiRequestFactory))
    , apiExecutor_(std::move(apiExecutor))
    , pushHandler_(std::move(pushHandler))
    , callingMessagesSender_(std::move(callingMessagesSender))
    , mediaSessionFactory_(std::move(mediaSessionFactory))
    , callHolder_(std::move(callHolder))
    , platformInfo_(std::move(platformInfo))
    , userGuid_(userGuid)
    , receiverDeviceId_(receiverDeviceId)
{
    Y_VERIFY(platformInfo_);
}

CallsController::~CallsController() = default;

void CallsController::startOutgoingCall(const std::string& callPayload) {
    if (!chatId_.empty()) {
        startOutgoingCallInNonPredictedChat(callPayload);
        return;
    }
    auto weakRef = weak_from(this);
    createPrivateChatTask_ = std::make_shared<PrepareChatApiTask>(
        apiRequestFactory_, userGuid_,
        [weakRef, callPayload](const Json::Value& chatInfo, const Json::Value& userInfo) {
            if (auto me = weakRef.lock()) {
                me->chatInfo_ = chatInfo;
                me->userInfo_ = userInfo;
                me->avatarUrl_ = userInfo["avatar_url"].asString();
                me->displayName_ = userInfo["display_name"].asString();
                me->chatId_ = me->chatInfo_["chat_id"].asString();
                me->startOutgoingCallInNonPredictedChat(callPayload);
            }
        },
        [weakRef](int code) {
            if (auto me = weakRef.lock()) {
                YIO_LOG_ERROR_EVENT("CallsController.CreateChatFailed", "Can't create chat");
                me->callHolder_->onCallCreationFailed.notifyObservers(
                    me->userGuid_,
                    code == 403
                        ? CallTransport::ErrorCode::BLOCKED_BY_PRIVACY_SETTINGS
                        : CallTransport::ErrorCode::UNKNOWN);
            }
        });
    apiExecutor_->start(createPrivateChatTask_);
}

void CallsController::startOutgoingCallInNonPredictedChat(const std::string& callPayload) {
    if (callHolder_->getCall()) {
        return;
    }
    call_ = buildOutgoingCall(callPayload);
    onEndSubscription_ =
        call_->notifier->subscribeOnEnd([this](std::string chatId) {
            if (chatId != call_->getChatId()) {
                return;
            }
            call_ = nullptr;
            callHolder_->setCall(nullptr);
        });
    callHolder_->setCall(call_);
    call_->start();
}

void CallsController::handleIncomingCall(proto::CallingMessage message, proto::ServerMessage serverMessage) {
    std::string callGuid = message->callguid();
    if (std::find(std::begin(previousCalls_), std::end(previousCalls_),
                  callGuid) != std::end(previousCalls_)) {
        YIO_LOG_INFO("Already handled callGuid="
                     << callGuid << ", probably delayed push message");
        return;
    }
    if (callHolder_->getCall()) {
        YIO_LOG_INFO("We already have a call");
        sendDeclineCall(callGuid);
        return;
    }
    if (const auto displayName = getDisplayName(serverMessage); displayName.has_value()) {
        displayName_ = *displayName;
    }
    if (const auto avatarUrl = getAvatarUrl(serverMessage);
        avatarUrl.has_value()) {
        avatarUrl_ = *avatarUrl;
    }
    fetchInfoForIncomingCall();
    call_ = buildIncomingCall(message);
    onEndSubscription_ =
        call_->notifier->subscribeOnEnd([this](std::string chatId) {
            if (chatId != call_->getChatId()) {
                return;
            }
            call_ = nullptr;
            callHolder_->setCall(nullptr);
        });

    callHolder_->setCall(call_);
    call_->start();
    previousCalls_.push_back(callGuid);
    if (previousCalls_.size() > MAX_CALL_GUID_COUNT) {
        previousCalls_.pop_front();
    }
}

void CallsController::fetchInfoForIncomingCall() {
    if (!chatInfo_.empty() && !userInfo_.empty()) {
        return;
    }
    if (fetchChatInfoTask_) {
        return;
    }
    auto weakRef = weak_from(this);
    fetchChatInfoTask_ = std::make_shared<PrepareChatApiTask>(
        apiRequestFactory_, userGuid_,
        [weakRef](const Json::Value& chatInfo, const Json::Value& userInfo) {
            if (auto me = weakRef.lock()) {
                me->chatInfo_ = chatInfo;
                me->userInfo_ = userInfo;
                me->avatarUrl_ = userInfo["avatar_url"].asString();
                me->displayName_ = userInfo["display_name"].asString();
                me->createPrivateChatTask_.reset();
                if (me->call_) {
                    bool changed = false;
                    changed |= me->call_->setUserName(me->displayName_);
                    changed |= me->call_->setUserAvatar(me->avatarUrl_);
                    if (changed) {
                        me->call_->notifier->notifyStatusChange();
                    }
                }
            }
        },
        [weakRef](int code) {
            YIO_LOG_ERROR_EVENT("CallsController.CreatePrivateChatTaskFailed", "CreatePrivateChatTask failed with code " << code);
        });
    apiExecutor_->start(fetchChatInfoTask_);
}

void CallsController::acceptIncomingCall() {
    if (call_) {
        call_->answer();
    }
}

void CallsController::declineIncomingCall() {
    if (call_) {
        call_->decline();
    }
}

void CallsController::hangupCall() {
    if (createPrivateChatTask_) {
        apiExecutor_->cancel(createPrivateChatTask_);
        createPrivateChatTask_.reset();
    }
    if (call_) {
        call_->stop();
    }
}

std::shared_ptr<Call> CallsController::buildOutgoingCall(const std::string& callPayload) {
    Y_VERIFY(!chatId_.empty());

    auto platformInfo = platformInfo_->create();
    deviceId_ = platformInfo.deviceId;
    auto deviceInfo = getDeviceInfo(platformInfo);
    deviceInfo->setCustomPayload(callPayload);

    auto messengerCallTransport = MessengerCallTransport::create(
        callingMessagesSender_, workerThread_,
        deviceId_, receiverDeviceId_,
        chatId_, std::string());

    auto mediaSession = mediaSessionFactory_->create(
        messengerCallTransport, workerThread_, deviceInfo, std::string());
    messengerCallTransport->initialize(mediaSession->getCallGuid());

    YIO_LOG_DEBUG("Passed DeviceInfo: " << deviceInfo->payload());
    return Call::create(chatId_, displayName_, avatarUrl_,
                        userGuid_, "", "", rtc::Direction::OUTGOING,
                        weak_from_this(), std::move(mediaSession), workerThread_,
                        messengerCallTransport);
}

std::shared_ptr<Call>
CallsController::buildIncomingCall(proto::CallingMessage message) {
    Y_VERIFY(message);
    Y_VERIFY(chatId_.empty() || chatId_ == message->chatid());

    chatId_ = message->chatid();
    auto platformInfo = platformInfo_->create();
    deviceId_ = platformInfo.deviceId;
    auto deviceInfo = getDeviceInfo(platformInfo);

    std::string callGuid = message->callguid();
    auto messengerCallTransport = MessengerCallTransport::create(
        callingMessagesSender_, workerThread_,
        deviceId_, receiverDeviceId_,
        chatId_, callGuid);

    messengerCallTransport->initialize(callGuid);

    auto mediaSession = mediaSessionFactory_->create(
        messengerCallTransport, workerThread_, deviceInfo, callGuid);

    YIO_LOG_DEBUG("Passed DeviceInfo: " << deviceInfo->payload());
    return Call::create(chatId_, displayName_, avatarUrl_,
                        userGuid_, message->deviceid(), message->incomingcall().payload(), rtc::Direction::INCOMING,
                        weak_from_this(), std::move(mediaSession), workerThread_,
                        messengerCallTransport);
}

void CallsController::sendDeclineCall(const std::string& callGuid) {
    auto callingMessage = proto::make<proto::CallingMessage>();
    callingMessage->set_deviceid(TString(deviceId_));
    callingMessage->set_chatid(TString(chatId_));
    callingMessage->set_callguid(TString(callGuid));
    callingMessage->mutable_declinecall();
    auto ignore = callingMessagesSender_->sendCallingMessage(quasar::makeUUID(),
                                                             callingMessage);
}
