#include "calls_controller_provider.h"

#include "calls_controller.h"
#include "call/call_holder.h"

#include <mssngr/router/lib/protos/client.pb.h>
#include <yandex_io/callkit/api/request_user_task.h>
#include <yandex_io/callkit/api/session_request_factory.h>
#include <yandex_io/callkit/connection/push_handler.h>
#include <yandex_io/callkit/connection/request_handler.h>
#include <yandex_io/callkit/rtc/media/media_session_factory.h>
#include <yandex_io/callkit/session/session_settings.h>
#include <yandex_io/callkit/storage/data_storage.h>
#include <yandex_io/callkit/util/async_service.h>
#include <yandex_io/callkit/util/loop_thread.h>

#include <yandex_io/libs/base/utils.h>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

namespace {

    const std::chrono::duration REQUEST_USER_MIN_INTERVAL = std::chrono::seconds(60);

    std::string userGuidFromChatId(const std::string& chatId,
                                   const std::string& ownGuid) {
        Y_VERIFY(quasar::isUUID(ownGuid));
        Y_VERIFY(chatId.size() == 73);

        if (chatId.substr(0, 36) == ownGuid) {
            return chatId.substr(37);
        }
        if (chatId.substr(37) == ownGuid) {
            return chatId.substr(0, 36);
        }
        return chatId;
    }

} // namespace

void CallsControllerProvider::init(
    const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory,
    const SessionSettings& settings,
    std::shared_ptr<LoopThread> workerThread,
    std::shared_ptr<DataStorage> dataStorage,
    std::shared_ptr<SessionRequestFactory> apiRequestFactory,
    std::shared_ptr<AsyncService> apiExecutor,
    std::shared_ptr<RequestHandler> requestHandler,
    std::shared_ptr<PushHandler> pushHandler,
    std::shared_ptr<CallingMessagesSender> callingMessagesSender,
    std::shared_ptr<CallHolder> callHolder,
    std::shared_ptr<quasar::IUserConfigProvider> userConfigProvider,
    Json::Value callsConfig) {
    workerThread_ = std::move(workerThread);
    dataStorage_ = std::move(dataStorage);
    apiRequestFactory_ = std::move(apiRequestFactory);
    apiService_ = std::move(apiExecutor);
    requestHandler_ = std::move(requestHandler);
    pushHandler_ = std::move(pushHandler);
    callingMessagesSender_ = std::move(callingMessagesSender);
    callHolder_ = std::move(callHolder);
    platformInfo_ = settings.params.platformInfo;
    deviceId_ = platformInfo_->create().deviceId;
    allowUsualCalls_ = false;
    mediaSessionFactory_ = std::make_shared<rtc::MediaSessionFactory>(ipcFactory, userConfigProvider, std::move(callsConfig));
    lastRequestUser_ = std::chrono::steady_clock::now() - REQUEST_USER_MIN_INTERVAL;

    if (!tryUpdateOwnGuid()) {
        apiService_->start(std::make_shared<RequestUserTask>(apiRequestFactory_, dataStorage_, settings.params.useDedicatedXiva));
        lastRequestUser_ = std::chrono::steady_clock::now();
    }

    messageSubscription_ = callingMessagesSender_->onCallingMessage.subscribe(
        this, [this, useDedicatedXiva = settings.params.useDedicatedXiva](proto::CallingMessage message, proto::ServerMessage serverMessage) {
            if (!tryUpdateOwnGuid()) {
                YIO_LOG_ERROR_EVENT("CallsControllerProvider.CallWithoutAuth", "Call received without auth data.");
                if (const auto now = std::chrono::steady_clock::now(); now - lastRequestUser_ >= REQUEST_USER_MIN_INTERVAL) {
                    apiService_->start(std::make_shared<RequestUserTask>(apiRequestFactory_, dataStorage_, useDedicatedXiva));
                    lastRequestUser_ = now;
                }
                return;
            }
            if (!message->has_incomingcall()) {
                return;
            }

            std::string callerGuid =
                userGuidFromChatId(message->chatid(), ownGuid_);

            std::string receiverDeviceId = message->receiverdeviceid();
            if (receiverDeviceId.empty()) {
                if (!allowUsualCalls_) {
                    YIO_LOG_INFO("Ignoring usual incoming call: " << message->ShortDebugString());
                    return;
                }
                YIO_LOG_INFO("Handling usual incoming call: " << message->ShortDebugString());
            } else {
                if (callerGuid != ownGuid_) {
                    // Calls on device from other users are impossible.
                    YIO_LOG_WARN("Device call from foreign user: " << message->ShortDebugString());
                    return;
                }
                if (receiverDeviceId != deviceId_) {
                    YIO_LOG_INFO("Ignoring foreign incoming call: " << message->ShortDebugString());
                    return;
                }
                YIO_LOG_INFO("Handling own device incoming call: " << message->ShortDebugString());
            }
            getByUserGuid(callerGuid)->handleIncomingCall(std::move(message), std::move(serverMessage));
        });
}

CallsControllerProvider::~CallsControllerProvider() = default;

void CallsControllerProvider::setAllowUsualCalls(bool allowUsualCalls) {
    allowUsualCalls_ = allowUsualCalls;
}

std::shared_ptr<CallsController> CallsControllerProvider::getByUserGuid(const std::string& userGuid) {
    if (!controllersForCalls_.contains(userGuid)) {
        controllersForCalls_[userGuid] = std::make_shared<CallsController>(
            workerThread_, apiRequestFactory_, apiService_,
            pushHandler_, callingMessagesSender_,
            mediaSessionFactory_, callHolder_, platformInfo_,
            userGuid);
    }
    return controllersForCalls_[userGuid];
}

std::shared_ptr<CallsController> CallsControllerProvider::getByDeviceId(const std::string& deviceId) {
    if (!controllersForCallsToOwnDevice_.contains(deviceId)) {
        controllersForCallsToOwnDevice_[deviceId] = std::make_shared<CallsController>(
            workerThread_, apiRequestFactory_, apiService_,
            pushHandler_, callingMessagesSender_,
            mediaSessionFactory_, callHolder_, platformInfo_,
            ownGuid_, deviceId);
    }
    return controllersForCallsToOwnDevice_[deviceId];
}

bool CallsControllerProvider::tryUpdateOwnGuid() {
    if (!ownGuid_.empty()) {
        return true;
    }
    auto user = dataStorage_->tryGetUser();
    if (user.isNull()) {
        return false;
    }
    if (!user.isMember("guid")) {
        return false;
    }
    ownGuid_ = user["guid"].asString();
    return !ownGuid_.empty();
}
