#include "messenger_call_transport.h"

#include "calling_message_receiver.h"

#include <mssngr/router/lib/protos/message.pb.h>
#include <yandex_io/callkit/rtc/media/device_info.h>
#include <yandex_io/callkit/util/loop_thread.h>

#include <yandex_io/libs/base/utils.h>

#include <algorithm>
#include <chrono>
#include <cstdint>

using namespace messenger;
using messenger::rtc::Transport;

namespace {
    std::chrono::milliseconds SEND_QUEUE_DISPOSE_DELAY(5000);
} // namespace

void MessengerCallTransport::init(
    std::shared_ptr<CallingMessagesSender> callingMessagesSender,
    std::shared_ptr<LoopThread> workerThread,
    const std::string& deviceId, const std::string& receiverDeviceId,
    const std::string& chatId, const std::string& callGuid) {
    callingMessagesSender_ = std::move(callingMessagesSender);
    workerThread_ = std::move(workerThread);
    deviceId_ = deviceId;
    receiverDeviceId_ = receiverDeviceId;
    chatId_ = chatId;
    sendSequenceNumber_ = 1u;

    Y_VERIFY(workerThread_->checkInside());

    uint64_t receiverSequenceNumber = 1u;
    if (!callGuid.empty()) {
        // Null callGuid implies an incoming call so the first message should be
        // skipped.
        receiverSequenceNumber = 2;
    }
    receiver_ = std::make_shared<CallingMessageReceiver>(
        listeners, workerThread_, deviceId_, receiverSequenceNumber);
    messageSubscription_ = callingMessagesSender_->onCallingMessage.subscribe(
        this, std::bind(&MessengerCallTransport::onMessageReceived, this,
                        std::placeholders::_1, std::placeholders::_2));
    ackSubscription_ = callingMessagesSender_->onAckReceived.subscribe(
        this, std::bind(&MessengerCallTransport::onAckReceived, this,
                        std::placeholders::_1));
    errorSubscription_ = callingMessagesSender_->onErrorReceived.subscribe(
        this, std::bind(&MessengerCallTransport::onErrorReceived, this,
                        std::placeholders::_1, std::placeholders::_2));
}

MessengerCallTransport::~MessengerCallTransport() {
    Y_VERIFY(workerThread_->checkInside());

    for (Cancelable& cancelable : allRequests_) {
        cancelable.cancel();
    }
    allRequests_.clear();
}

void MessengerCallTransport::initialize(const std::string& callGuid) {
    Y_VERIFY(workerThread_->checkInside());
    callGuid_ = callGuid;
    receiver_->setCallGuid(callGuid_);
}

void MessengerCallTransport::dispose() {
    Y_VERIFY(workerThread_->checkInside());

    YIO_LOG_INFO("Disposing MessengerCallTransport: " << this);

    messageSubscription_.reset();
    ackSubscription_.reset();
    errorSubscription_.reset();

    receiver_->dispose();
    // We should poll head element from the queue since we already put it in
    // SocketConnection through CallingMessagesSender#sendCallingMessage() in
    // trySend() method before. Otherwise, duplicates will occur.
    if (!sendQueue_.empty()) {
        sendQueue_.pop_front();
    }
    flushQueue();
    workerThread_->executeDelayed(bindWeak(this, [this] { delayedDispose(); }),
                                  SEND_QUEUE_DISPOSE_DELAY);
}

void MessengerCallTransport::delayedDispose() {
    Y_VERIFY(workerThread_->checkInside());

    for (Cancelable& cancelable : allRequests_) {
        cancelable.cancel();
    }
    allRequests_.clear();
}

void MessengerCallTransport::send(const rtc::Transport::Message& message) {
    Y_VERIFY(workerThread_->checkInside());
    enqueueMessage(buildMediaTransportMessage(message));
}

void MessengerCallTransport::setListener(std::weak_ptr<Listener> listener) {
    receiver_->setMediaTransportListener(std::move(listener));
}

// CallTransport
RequestId
MessengerCallTransport::makeCall(std::shared_ptr<rtc::DeviceInfo> deviceInfo) {
    Y_VERIFY(workerThread_->checkInside());
    return enqueueMessage(buildMakeCallMessage(deviceInfo));
}

RequestId MessengerCallTransport::acceptCall(
    std::shared_ptr<rtc::DeviceInfo> deviceInfo) {
    Y_VERIFY(workerThread_->checkInside());
    return enqueueMessage(buildAcceptCallMessage(deviceInfo));
}

RequestId MessengerCallTransport::declineCall() {
    Y_VERIFY(workerThread_->checkInside());
    return enqueueMessage(buildDeclineCallMessage());
}

RequestId MessengerCallTransport::endCall() {
    Y_VERIFY(workerThread_->checkInside());
    return enqueueMessage(buildEndCallMessage());
}

RequestId MessengerCallTransport::notifyRinging() {
    Y_VERIFY(workerThread_->checkInside());
    auto callingMessage = buildCallingMessage();
    callingMessage->mutable_notifyringing();
    return enqueueMessage(std::move(callingMessage));
}

RequestId
MessengerCallTransport::enqueueMessage(proto::CallingMessage callingMessage) {
    auto requestId = RequestId::generate();
#ifndef _NDEBUG
    YIO_LOG_DEBUG("enqueueMessage requestId="
                  << requestId.getId()
                  << " message=" << callingMessage->ShortDebugString());
#else
    YIO_LOG_DEBUG("enqueueMessage requestId=" << requestId.getId());
#endif
    bool wasEmpty = sendQueue_.empty();
    sendQueue_.push_back({requestId, callingMessage});
    if (wasEmpty) {
        trySend();
    }
    sendSequenceNumber_++;
    return requestId;
}

bool MessengerCallTransport::dequeueMessage(const RequestId& requestId) {
    auto it = std::find_if(std::begin(sendQueue_), std::end(sendQueue_),
                           [&requestId](const QueueMessage& pair) {
                               return pair.first == requestId;
                           });
    if (it != std::end(sendQueue_)) {
        sendQueue_.erase(it);
        return true;
    }
    return false;
}

void MessengerCallTransport::flushQueue() {
    for (auto pair : sendQueue_) {
        Y_VERIFY(pair.first);
        Y_VERIFY(pair.second);

        auto cancelable = callingMessagesSender_->sendCallingMessage(
            pair.first.getId(), pair.second);
        allRequests_.push_back(cancelable);
    }
    quasar::clear(sendQueue_);
}

void MessengerCallTransport::trySend() {
    if (sendQueue_.empty()) {
        return;
    }
    auto pair = sendQueue_.front();
    Y_VERIFY(pair.first);
    Y_VERIFY(pair.second);

    auto cancelable = callingMessagesSender_->sendCallingMessage(
        pair.first.getId(), pair.second);
    allRequests_.push_back(cancelable);
}

proto::CallingMessage MessengerCallTransport::buildMakeCallMessage(
    std::shared_ptr<rtc::DeviceInfo> deviceInfo) {
    proto::CallingMessage callingMessage = buildCallingMessage();
    callingMessage->mutable_makecall()->set_deviceinfo(deviceInfo->payload());
    callingMessage->mutable_makecall()->set_calltype(
        ::NMessengerProtocol::ECallType::CALL_TYPE_AUDIO);
    return callingMessage;
}

proto::CallingMessage MessengerCallTransport::buildMediaTransportMessage(
    const rtc::Transport::Message& message) {
    proto::CallingMessage callingMessage = buildCallingMessage();
    callingMessage->mutable_transportmessage()->set_payload(TString(message));
    return callingMessage;
}

proto::CallingMessage MessengerCallTransport::buildAcceptCallMessage(
    std::shared_ptr<rtc::DeviceInfo> deviceInfo) {
    proto::CallingMessage callingMessage = buildCallingMessage();
    callingMessage->mutable_acceptcall()->set_deviceinfo(deviceInfo->payload());
    return callingMessage;
}

proto::CallingMessage MessengerCallTransport::buildDeclineCallMessage() {
    proto::CallingMessage callingMessage = buildCallingMessage();
    callingMessage->mutable_declinecall();
    return callingMessage;
}

proto::CallingMessage MessengerCallTransport::buildEndCallMessage() {
    proto::CallingMessage callingMessage = buildCallingMessage();
    callingMessage->mutable_endcall();
    return callingMessage;
}

proto::CallingMessage MessengerCallTransport::buildCallingMessage() {
    proto::CallingMessage callingMessage = proto::make<proto::CallingMessage>();
    callingMessage->set_deviceid(TString(deviceId_));
    callingMessage->set_chatid(TString(chatId_));
    callingMessage->set_callguid(TString(callGuid_));
    callingMessage->set_seqno(sendSequenceNumber_);

    if (!receiverDeviceId_.empty()) {
        callingMessage->set_receiverdeviceid(TString(receiverDeviceId_));
    }
    return callingMessage;
}

void MessengerCallTransport::onMessageReceived(proto::CallingMessage message, proto::ServerMessage /*serverMessage*/) {
    Y_VERIFY(workerThread_->checkInside());

    // We filter out calling messages by callGuid and by chat. Because of cases
    // when a user calls himself due to the redirection: in one chat call is
    // outgoing and in other is incoming. For example, chats with Health and
    // business.
    if (callGuid_ != message->callguid() || chatId_ != message->chatid()) {
        return;
    }
    // 'Call to my device' produces mirrored messages to other devices that must
    // be cut, see QUASAR-7366
    if (!message->receiverdeviceid().empty() &&
        message->receiverdeviceid() != deviceId_) {
        return;
    }
    receiver_->onCallingMessage(message);
}

void MessengerCallTransport::onAckReceived(const std::string& payloadId) {
    YIO_LOG_DEBUG("onAckReceived(" + payloadId + ")");

    RequestId requestId = RequestId(payloadId);
    bool removed = dequeueMessage(requestId);
    if (removed) {
        trySend();
    }

    listeners->onAckReceived.notifyObservers(requestId);
}

void MessengerCallTransport::onErrorReceived(
    const std::string& payloadId, proto::PostMessageResponse response) {
    YIO_LOG_WARN("onErrorReceived(payloadId=" + payloadId + ")");

    RequestId requestId = RequestId(payloadId);
    bool removed = dequeueMessage(requestId);
    if (removed) {
        trySend();
    }

    ErrorCode errorCode = ErrorCode::UNKNOWN;
    if (response) {
        auto status = response->status();
        switch ((proto::CommitStatus)status) {
            case proto::CommitStatus::CONFLICT:
                errorCode = ErrorCode::CONFLICT;
                break;
            case proto::CommitStatus::BAD_REQUEST:
                errorCode = ErrorCode::BAD_REQUEST;
                break;
            case proto::CommitStatus::BLOCKED_BY_PRIVACY_SETTINGS:
                errorCode = ErrorCode::BLOCKED_BY_PRIVACY_SETTINGS;
                break;
            default:
                break;
        }
    }
    listeners->onErrorReceived.notifyObservers(requestId, errorCode);
}
