#pragma once

#include <google/protobuf/message.h>

#include <memory>
#include <string>
#include <type_traits>

namespace messenger {

    struct BaseRequestTask {
        enum Type {
            MESSAGE, // only send message, no result
            REQUEST, // message sent, result must be received
        };

        virtual ~BaseRequestTask() = default;

        virtual google::protobuf::Message* createRequestMessage() = 0;
        virtual Type getType() = 0;
        virtual std::string getPath() = 0;
        // Return true if the result contains a error and the request
        // should be retried.
        virtual bool
        onResult(std::shared_ptr<google::protobuf::Message> message) = 0;
        virtual void onError() = 0;

        // Required for Type::REQUEST
        virtual google::protobuf::Message* createResultMessage() = 0;

        // For debug and logging
        virtual std::string getName() = 0;
    };

    template <typename RequestType, typename ResultType>
    class RequestTask: public BaseRequestTask {
        static_assert(
            std::is_base_of<google::protobuf::Message, RequestType>::value,
            "Not a proto message");
        static_assert(std::is_base_of<google::protobuf::Message, ResultType>::value,
                      "Not a proto message");

    public:
        ~RequestTask() override = default;

        virtual void prepareRequest(RequestType* message) = 0;
        virtual bool onResultMessage(std::shared_ptr<ResultType> message) = 0;

    private:
        Type getType() final;
        bool onResult(std::shared_ptr<google::protobuf::Message> message) final;
        google::protobuf::Message* createRequestMessage() final;
        // Required for Type::REQUEST
        google::protobuf::Message* createResultMessage() final;
    };

    template <typename RequestType>
    class MessageTask: public BaseRequestTask {
        static_assert(
            std::is_base_of<google::protobuf::Message, RequestType>::value,
            "Not a proto message");

    public:
        ~MessageTask() override = default;
        virtual void prepareRequest(RequestType* message) = 0;

    private:
        std::string getPath() final;
        Type getType() final;
        bool onResult(std::shared_ptr<google::protobuf::Message> message) final;
        google::protobuf::Message* createRequestMessage() final;
        // Required for Type::REQUEST
        google::protobuf::Message* createResultMessage() final;
    };

    template <typename RequestType, typename ResultType>
    BaseRequestTask::Type RequestTask<RequestType, ResultType>::getType() {
        return REQUEST;
    }

    template <typename RequestType, typename ResultType>
    bool RequestTask<RequestType, ResultType>::onResult(
        std::shared_ptr<google::protobuf::Message> message) {
        return onResultMessage(std::static_pointer_cast<ResultType>(message));
    }

    template <typename RequestType, typename ResultType>
    google::protobuf::Message*
    RequestTask<RequestType, ResultType>::createRequestMessage() {
        auto* message = new RequestType();
        prepareRequest(message);
        return message;
    }

    template <typename RequestType, typename ResultType>
    google::protobuf::Message*
    RequestTask<RequestType, ResultType>::createResultMessage() {
        return new ResultType();
    }

    template <typename RequestType>
    std::string MessageTask<RequestType>::getPath() {
        return "push";
    }

    template <typename RequestType>
    BaseRequestTask::Type MessageTask<RequestType>::getType() {
        return MESSAGE;
    }

    template <typename RequestType>
    bool MessageTask<RequestType>::onResult(
        std::shared_ptr<google::protobuf::Message> message) {
        Y_VERIFY(!"Not reached");
        return false;
    }

    template <typename RequestType>
    google::protobuf::Message* MessageTask<RequestType>::createRequestMessage() {
        auto* message = new RequestType();
        prepareRequest(message);
        return message;
    }

    template <typename RequestType>
    google::protobuf::Message* MessageTask<RequestType>::createResultMessage() {
        Y_VERIFY(!"Not reached");
        return nullptr;
    }

} // namespace messenger
