#include "connection_service.h"

#include "base_request_task.h"
#include "client_request_task.h"
#include "dedicated_socket.h"
#include "push_handler.h"
#include "pushd_socket.h"
#include "request_handler.h"

#include <mssngr/router/lib/protos/client.pb.h>
#include <mssngr/router/lib/protos/message.pb.h>
#include <yandex_io/callkit/storage/data_storage.h>
#include <yandex_io/callkit/util/loop_thread.h>
#include <yandex_io/callkit/xiva/transport.h>

#include <yandex_io/libs/logging/logging.h>

#include <json/value.h>

#include <string>

using namespace messenger;

namespace {

    class HeartbeatTask: public ClientRequestTask {
    public:
        HeartbeatTask() {
        }

        std::string getName() override {
            return "HeartbeatTask";
        }

        void prepareClientMessage(
            NMessengerProtocol::TClientMessage* clientMessage) override {
            clientMessage->mutable_heartbeat()->set_onlineuntil(0u);
        }
    };

} // namespace

ConnectionService::ConnectionService(const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory,
                                     const SessionSettings& settings,
                                     std::shared_ptr<LoopThread> workerThread,
                                     std::shared_ptr<DataStorage> dataStorage,
                                     std::shared_ptr<YandexIO::ITelemetry> telemetry)
    : destroyed_(false)
    , settings_(settings)
    , workerThread_(std::move(workerThread))
    , dataStorage_(std::move(dataStorage))
{
    if (settings.params.useDedicatedXiva) {
        socket_ = std::make_shared<DedicatedSocket>(settings_, dataStorage_, std::move(telemetry));
    } else {
        socket_ = std::make_shared<PushdSocket>(ipcFactory, settings_.params.xivaSubscriptions);
    }
    auto xivaTransport = std::make_shared<xiva::Transport>(
        socket_, settings_.params.xivaServiceName);
    requestHandler_ =
        std::make_shared<RequestHandler>(workerThread_, xivaTransport);
    pushHandler_ = std::make_shared<PushHandler>(workerThread_, xivaTransport);
}

ConnectionService::~ConnectionService() {
    YIO_LOG_INFO("ConnectionService destroy");
    Y_VERIFY(!destroyed_);
    destroyed_ = true;
}

bool ConnectionService::isConnected() const {
    return socket_->getStatus() == xiva::SocketApi::Status::CONNECTED;
}

void ConnectionService::sendHeartbeat() {
    processTask(std::make_shared<HeartbeatTask>());
}

bool ConnectionService::hasBgTasks() const {
    Y_VERIFY(!destroyed_);
    if (destroyed_) {
        return false;
    }
    return requestHandler_->hasBgTasks();
}

void ConnectionService::runOnBgIdle(std::function<void()> callback) {
    Y_VERIFY(!destroyed_);
    if (destroyed_) {
        return;
    }
    requestHandler_->runOnBgIdle(std::move(callback));
}

std::shared_ptr<PushHandler> ConnectionService::getPushHandler() const {
    Y_VERIFY(!destroyed_);
    return pushHandler_;
}

std::shared_ptr<RequestHandler> ConnectionService::getRequestHandler() const {
    Y_VERIFY(!destroyed_);
    return requestHandler_;
}

void ConnectionService::processTask(std::shared_ptr<BaseRequestTask> task) {
    Y_VERIFY(!destroyed_);
    if (destroyed_) {
        return;
    }
    requestHandler_->process(std::move(task));
}
