#include "dedicated_socket.h"

#include <yandex_io/callkit/storage/data_storage.h>

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/websocket/websocket_client.h>

#include <json/value.h>

#include <sstream>

using namespace messenger;

DedicatedSocket::DedicatedSocket(const SessionSettings& settings,
                                 std::shared_ptr<DataStorage> dataStorage,
                                 std::shared_ptr<YandexIO::ITelemetry> telemetry)
    : settings_(settings)
    , dataStorage_(dataStorage)
    , webClient_(std::make_unique<quasar::WebsocketClient>(std::move(telemetry)))
{
    Json::Value xivaSecret = dataStorage_->tryGetXivaSecret();
    if (xivaSecret.isNull()) {
        setupXivaSubscription_ = dataStorage_->subscribeUserData([this] {
            Json::Value xivaSecret = dataStorage_->tryGetXivaSecret();
            if (quasar::tryGetString(xivaSecret, "sign", "").empty() ||
                quasar::tryGetString(xivaSecret, "ts", "").empty()) {
                return;
            }
            settings_.xivaSettings.sign = xivaSecret["sign"].asString();
            settings_.xivaSettings.ts = xivaSecret["ts"].asString();
            setupXivaSubscription_.reset();
            connect();
        });
    } else {
        settings_.xivaSettings.sign = xivaSecret["sign"].asString();
        settings_.xivaSettings.ts = xivaSecret["ts"].asString();
        connect();
    }
}

DedicatedSocket::~DedicatedSocket() = default;

std::vector<std::string> DedicatedSocket::getServiceNames() {
    return {settings_.params.xivaServiceName};
}

void DedicatedSocket::sendBinaryData(const std::string& buffer,
                                     SocketApi::FailCallback onFail) {
    if (getStatus() != Status::CONNECTED) {
        callbackQueue_.add([onFail] { onFail(); });
        return;
    }
    webClient_->unsafeSendBinary(buffer);
}

void DedicatedSocket::sendTextData(const std::string& buffer,
                                   SocketApi::FailCallback onFail) {
    if (getStatus() != Status::CONNECTED) {
        callbackQueue_.add([onFail] { onFail(); });
        return;
    }
    webClient_->unsafeSend(buffer);
}

void DedicatedSocket::connect() {
    handleStatus(Status::CONNECTING);
    quasar::WebsocketClient::Settings wsSettings;
    std::stringstream url;
    url << "wss://push.yandex.ru/v2/subscribe/websocket?"
        << "&service=" << settings_.params.xivaServiceName << "&client=q"
        << "&session=" << settings_.sessionId
        << "&ts=" << settings_.xivaSettings.ts
        << "&user=" << settings_.params.passportUid;
    if (!settings_.xivaSettings.sign.empty() &&
        !settings_.xivaSettings.ts.empty()) {
        YIO_LOG_DEBUG("Connecting with sign");
        url << "&sign=" << settings_.xivaSettings.sign
            << "&ts=" << settings_.xivaSettings.ts;
    } else {
        YIO_LOG_DEBUG("Connecting with oauth");
        wsSettings.connectionHeaders["Authorization"] =
            "OAuth " + settings_.params.token;
    }
    wsSettings.url = url.str();
    wsSettings.tls.crtFilePath = "/system/vendor/quasar/ca-certificates.crt";
    webClient_->setOnMessageHandler(std::bind(&DedicatedSocket::onTextMessage,
                                              this, std::placeholders::_1));
    webClient_->setOnBinaryMessageHandler(std::bind(
        &DedicatedSocket::onBinaryMessage, this, std::placeholders::_1));
    webClient_->setOnConnectHandler(
        std::bind(&DedicatedSocket::onConnected, this));
    webClient_->setOnDisconnectHandler(std::bind(
        &DedicatedSocket::onDisconnected, this, std::placeholders::_1));
    webClient_->setOnFailHandler(
        std::bind(&DedicatedSocket::onFail, this, std::placeholders::_1));
    YIO_LOG_DEBUG("Connecting with url " << wsSettings.url);
    webClient_->connectAsync(wsSettings);
}

void DedicatedSocket::disconnect() {
    webClient_->disconnectAsync();
}

void DedicatedSocket::onTextMessage(const std::string& msg) {
    YIO_LOG_DEBUG("onTextMessage: " + msg);
    handleTextData(msg);
}

void DedicatedSocket::onBinaryMessage(const std::string& msg) {
    YIO_LOG_DEBUG("onBinaryMessage: " + std::to_string(msg.size()));
    handleBinaryData(msg);
}

void DedicatedSocket::onConnected() {
    YIO_LOG_DEBUG("onConnected");
    handleStatus(Status::CONNECTED);
}

void DedicatedSocket::onDisconnected(
    const quasar::Websocket::ConnectionInfo& info) {
    YIO_LOG_DEBUG("onDisconnected");
    disconnect();
    handleStatus(Status::DISCONNECTED);
}

void DedicatedSocket::onFail(const quasar::Websocket::ConnectionInfo& info) {
    YIO_LOG_DEBUG("onFail");
    disconnect();
    handleStatus(Status::DISCONNECTED);
}
