#include "messenger_protocol.h"

#include <yandex_io/callkit/util/murmur_hash2.h>

#include <yandex_io/libs/logging/logging.h>

#include <sstream>
#include <utility>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

namespace {

    const uint32_t HEADER_VERSION = 4u;

    inline void writeUInt8(std::stringstream* out, uint8_t value) {
        out->put(*reinterpret_cast<char*>(&value));
    }

    inline void writeUInt32Le(std::stringstream* out, uint32_t value) {
        writeUInt8(out, value & 0xFFu);
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
    }

    inline void writeUInt64Le(std::stringstream* out, uint64_t value) {
        writeUInt8(out, value & 0xFFu);
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
        writeUInt8(out, (value >>= 8u, value & 0xFFu));
    }

    inline uint8_t readUInt8(std::stringstream* in) {
        char value;
        in->get(value);
        return *reinterpret_cast<uint8_t*>(&value);
    }

    inline uint32_t readUInt32Le(std::stringstream* in) {
        uint32_t value = readUInt8(in);
        value |= (((uint32_t)readUInt8(in)) << 8u);
        value |= (((uint32_t)readUInt8(in)) << 16u);
        value |= (((uint32_t)readUInt8(in)) << 24u);
        return value;
    }

    inline uint64_t readUInt64Le(std::stringstream* in) {
        uint64_t value = readUInt8(in);
        value |= (((uint32_t)readUInt8(in)) << 8u);
        value |= (((uint32_t)readUInt8(in)) << 16u);
        value |= (((uint32_t)readUInt8(in)) << 24u);
        value |= (((uint64_t)readUInt8(in)) << 32u);
        value |= (((uint64_t)readUInt8(in)) << 40u);
        value |= (((uint64_t)readUInt8(in)) << 48u);
        value |= (((uint64_t)readUInt8(in)) << 56u);
        return value;
    }

} // namespace

std::string messenger::extractMessagePayload(const std::string& payloadFromDataFrame) {
    std::stringstream buffer(payloadFromDataFrame);
    uint32_t headerVersion = readUInt32Le(&buffer);
    if (headerVersion != HEADER_VERSION) {
        YIO_LOG_INFO("onDataFrame: unknown header version " << headerVersion);
        return std::string();
    }
    uint64_t hash = readUInt64Le(&buffer);
    std::string payload(std::istreambuf_iterator<char>(buffer), {});
    uint64_t payloadHash = murmurHash2(payload);
    if (payloadHash != hash) {
        YIO_LOG_ERROR_EVENT("MessengerProtocol.OnDataFrame.InvalidCheckSum", "onDataFrame: invalid check sum: "
                                                                                 << std::to_string(hash) << " expected:" << payloadHash);
        return std::string();
    }
    return payload;
}

std::string messenger::extractPushPayload(const std::string& payloadFromDataFrame) {
    return extractMessagePayload(payloadFromDataFrame);
}

std::string messenger::createDataFramePayload(const std::string& messagePayload) {
    std::stringstream buffer;
    uint64_t hash = murmurHash2(messagePayload);
    writeUInt32Le(&buffer, HEADER_VERSION);
    writeUInt64Le(&buffer, hash);
    buffer << messagePayload;
    return buffer.str();
}
