#include "pushd_socket.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

YIO_DEFINE_LOG_MODULE("callkit");

using quasar::proto::QuasarMessage;
using quasar::proto::WebsocketClientState;

using namespace messenger;

PushdSocket::PushdSocket(const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory, const std::vector<std::string>& xivaSubscriptions)
    : xivaSubscriptions_(xivaSubscriptions)
    , toPushd_(ipcFactory->createIpcConnector("pushd"))
{
    toPushd_->setMessageHandler(std::bind(&PushdSocket::handleQuasarMessage, this, std::placeholders::_1));
    toPushd_->setConnectHandler(std::bind(&PushdSocket::subscribeToExtraPushdData, this));
    toPushd_->connectToService();

    toPushd_->waitUntilConnected(std::chrono::seconds(10));
}

PushdSocket::~PushdSocket() {
    toPushd_->shutdown();
}

std::vector<std::string> PushdSocket::getServiceNames() {
    return xivaSubscriptions_;
}

void PushdSocket::sendBinaryData(const std::string& buffer,
                                 PushdSocket::FailCallback onFail) {
    QuasarMessage message;
    *message.mutable_xiva_websocket_send_binary_message() = buffer;
    toPushd_->sendMessage(std::move(message));
}

void PushdSocket::sendTextData(const std::string& buffer,
                               PushdSocket::FailCallback onFail) {
    throw std::runtime_error("Not implemented");
}

void PushdSocket::handleQuasarMessage(const quasar::ipc::SharedMessage& message) {
    if (message->has_xiva_websocket_recv_binary_message()) {
        handleBinaryData(message->xiva_websocket_recv_binary_message());
    }

    if (message->has_xiva_websocket_client_state()) {
        switch (message->xiva_websocket_client_state().state()) {
            case WebsocketClientState::DISCONNECTED: {
                handleStatus(xiva::SocketApi::Status::DISCONNECTED);
                break;
            }
            case WebsocketClientState::CONNECTING: {
                handleStatus(xiva::SocketApi::Status::CONNECTING);
                break;
            }
            case WebsocketClientState::CONNECTED: {
                handleStatus(xiva::SocketApi::Status::CONNECTED);
                break;
            }
            default: {
                YIO_LOG_ERROR_EVENT("PushdSocket.InvalidWebsocketStatus", "Unknown websocket client status");
                break;
            }
        }
    }
}

void PushdSocket::subscribeToExtraPushdData() {
    QuasarMessage message;
    message.mutable_subscribe_to_pushd_xiva_websocket_events();
    toPushd_->sendMessage(std::move(message));
}
