#include "request_handler.h"

#include "base_request_task.h"
#include "messenger_protocol.h"

#include <mssngr/router/lib/protos/client.pb.h>
#include <yandex_io/callkit/util/loop_thread.h>
#include <yandex_io/callkit/util/weak_utils.h>
#include <yandex_io/callkit/xiva/request.h>
#include <yandex_io/callkit/xiva/transport.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

class messenger::XivaRequest
    : public xiva::Request,
      public std::enable_shared_from_this<messenger::XivaRequest> {
public:
    XivaRequest(std::shared_ptr<BaseRequestTask> task,
                std::shared_ptr<LoopThread> workerThread,
                std::weak_ptr<RequestHandler> requestHandler)
        : task_(std::move(task))
        , workerThread_(std::move(workerThread))
        , requestHandler_(requestHandler)
        , retryCount_(3)
        , retryDelayMillis_(2000)
    {
    }

    ~XivaRequest() override = default;

    [[nodiscard]] std::string getPayload() const override {
        auto message = std::shared_ptr<google::protobuf::Message>(
            task_->createRequestMessage());
        TString proto;
        Y_PROTOBUF_SUPPRESS_NODISCARD message->SerializeToString(&proto);
        YIO_LOG_INFO("Sending data with size: " << proto.size());
#ifndef _NDEBUG // avoid personal message data in logs?
        YIO_LOG_DEBUG(message->ShortDebugString());
#endif
        return createDataFramePayload(proto);
    }

    void onFail() override {
        YIO_LOG_ERROR_EVENT("XivaRequest.OnFail", "onFail: " + task_->getName());
        submitError();
    }

    void onDataFrame(const std::string& payload) override {
        YIO_LOG_DEBUG("onDataFrame: " + task_->getName() + ", payload size " +
                      std::to_string(payload.size()));
        auto type = task_->getType();
        if (type == BaseRequestTask::MESSAGE) {
            // answer is not required
            return;
        } else if (type == BaseRequestTask::REQUEST) {
            TString proto = extractMessagePayload(payload);
            if (proto.empty()) {
                submitError();
                return;
            }
            auto message = std::shared_ptr<google::protobuf::Message>(
                task_->createResultMessage());
            Y_PROTOBUF_SUPPRESS_NODISCARD message->ParseFromString(proto);
            YIO_LOG_INFO("Received data with size: " << proto.size());
#ifndef _NDEBUG // avoid personal message data in logs?
            YIO_LOG_DEBUG(message->ShortDebugString());
#endif
            submitResult(message);
        } else {
            Y_VERIFY(!"Type not handled");
        }
    }

    [[nodiscard]] std::string getPath() const override {
        return task_->getPath();
    }

    void onProxyStatus(uint16_t errorCode,
                       const std::string& errorName) override {
        YIO_LOG_ERROR_EVENT("XivaRequest.ProxyStatusError", "onProxyStatus: " + task_->getName() + ": " + errorName +
                                                                " " + std::to_string(errorCode));
        submitError();
    }

    void onTimeout() override {
        YIO_LOG_ERROR_EVENT("XivaRequest.Timeout", "onTimeout: " + task_->getName());
        submitError();
    }

    void onCancel() override {
        YIO_LOG_DEBUG("onCancel: " + task_->getName());
        submitCancel();
    }

private:
    friend class RequestHandler;

    void submitResult(std::shared_ptr<google::protobuf::Message> message) {
        // we have to handle additional racing cases since xiva::Transport does
        // not follow LooperThread convention.
        workerThread_->execute(bindWeak(this, requestHandler_, [this, message] {
            YIO_LOG_DEBUG("onResult: " + task_->getName());
            bool retryError = task_->onResult(message);
            if (retryError) {
                retryImpl();
            } else {
                requestHandler_.lock()->onRequestDone(requestId_);
            }
        }));
    }

    void submitError() {
        workerThread_->execute(
            bindWeak(this, requestHandler_, [this] { retryImpl(); }));
    }

    // on: worker thread
    void retryImpl() {
        Y_VERIFY(!requestHandler_.expired());
        if (retryCount_--) {
            YIO_LOG_ERROR_EVENT("XivaRequest.Retry", "Trying to retry: " + task_->getName());
            requestHandler_.lock()->onRequestRetry(requestId_,
                                                   retryDelayMillis_);
        } else {
            YIO_LOG_ERROR_EVENT("XivaRequest.Failed", "Task failed: " + task_->getName());
            task_->onError();
            requestHandler_.lock()->onRequestDone(requestId_);
        }
        retryDelayMillis_ *= 2;
    }

    void submitCancel() {
        workerThread_->execute(bindWeak(this, requestHandler_, [this] {
            requestHandler_.lock()->onRequestDone(requestId_);
        }));
    }

    uint32_t requestId_ = 0u;

    const std::shared_ptr<BaseRequestTask> task_;
    const std::shared_ptr<LoopThread> workerThread_;
    const std::weak_ptr<RequestHandler> requestHandler_;

    unsigned retryCount_;
    std::chrono::milliseconds retryDelayMillis_;
};

RequestHandler::RequestHandler(std::shared_ptr<LoopThread> workerThread,
                               std::shared_ptr<xiva::Transport> xivaTransport)
    : workerThread_(std::move(workerThread))
    , xivaTransport_(std::move(xivaTransport))
{
}

RequestHandler::~RequestHandler() {
    cancelActiveRequests();
    quasar::clear(idleCallbacks_);
}

Cancelable RequestHandler::process(std::shared_ptr<BaseRequestTask> task) {
    std::weak_ptr<RequestHandler> weakRef = shared_from_this();
    auto request = std::make_shared<XivaRequest>(task, workerThread_, weakRef);
    uint32_t requestId = startRequest(std::move(request));
    if (requestId != 0u) {
        return bindWeak(weakRef, std::bind([this, requestId] {
                            xivaTransport_->cancel(requestId);
                        }));
    }
    return Cancelable();
}

uint32_t RequestHandler::startRequest(std::shared_ptr<XivaRequest> request) {
    uint32_t requestId = xivaTransport_->send(request);
    if (requestId != 0u &&
        request->task_->getType() != BaseRequestTask::MESSAGE) {
        request->requestId_ = requestId;
        activeTasks_[requestId] = request;
        YIO_LOG_DEBUG("Request started with id: " << requestId);
    }
    return requestId;
}

std::shared_ptr<XivaRequest> RequestHandler::finishRequest(uint32_t requestId) {
    auto it = activeTasks_.find(requestId);
    if (it == activeTasks_.end()) {
        YIO_LOG_WARN("onRequestDone: no active request with id=" +
                     std::to_string(requestId));
        return std::shared_ptr<XivaRequest>();
    }
    auto request = it->second;
    activeTasks_.erase(it);
    request->requestId_ = 0u;
    YIO_LOG_DEBUG("Request finished with id: " << requestId);
    return request;
}

void RequestHandler::onRequestDone(uint32_t requestId) {
    if (requestId == 0u) {
        return;
    }
    if (!finishRequest(requestId)) {
        return;
    }
    processIdleState();
}

void RequestHandler::onRequestRetry(uint32_t requestId,
                                    std::chrono::milliseconds delay) {
    if (requestId == 0u) {
        return;
    }
    auto request = finishRequest(requestId);
    if (!request) {
        return;
    }
    retryTasks_.insert(request);
    workerThread_->executeDelayed(
        bindWeak(this, request,
                 [this, request] {
                     if (retryTasks_.erase(request)) {
                         YIO_LOG_ERROR_EVENT("RequestHandler.Retry", "Retry: " + request->task_->getName());
                         startRequest(request);
                     }
                 }),
        delay);
}

bool RequestHandler::hasBgTasks() const {
    return !activeTasks_.empty() || !retryTasks_.empty();
}

void RequestHandler::runOnBgIdle(std::function<void()> callback) {
    idleCallbacks_.push(std::move(callback));
}

void RequestHandler::stopAll() {
    cancelActiveRequests();
    processIdleState();
}

void RequestHandler::cancelActiveRequests() {
    xivaTransport_->cancelAll();
    activeTasks_.clear();
}

void RequestHandler::processIdleState() {
    if (!hasBgTasks()) {
        while (!idleCallbacks_.empty()) {
            auto callback = idleCallbacks_.front();
            idleCallbacks_.pop();
            if (callback) {
                callback();
            }
        }
    }
}
