#include "direct_tool_processor.h"

#include <yandex_io/libs/logging/logging.h>

#include <map>
#include <memory>
#include <ostream>
#include <stdexcept>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

struct ExitScriptSignal {};

struct Value {
    Value(const Value&) = default;
    Value(const std::string& value)
        : value(value)
    {
    }
    Value(const char* value)
        : Value(std::string(value))
    {
    }
    Value(bool value)
        : value(value ? "true" : "false")
    {
    }
    Value(int value)
        : value(std::to_string(value))
    {
    }
    Value(unsigned value)
        : value(std::to_string(value))
    {
    }
    operator bool() const {
        return value == "true" || value == "1" || value == "True" ||
               value == "TRUE" || value == "Yeah!" || value == "Yes" ||
               value == "yes";
    }
    operator std::string() const {
        return value;
    }
    operator int() const {
        return std::stoi(value, nullptr, 0);
    }
    operator unsigned() const {
        int i = *this;
        if (i < 0)
            throw std::runtime_error("Expected unsigned, got " + value);
        return (unsigned)i;
    }
    std::string value;
};

struct CommandData {
    CommandData(std::string line, size_t lineNumber)
        : lineNumber(lineNumber)
    {
        std::stringstream stream(line);
        std::string token;
        while (std::getline(stream, token, ' ')) {
            if (token.empty()) {
                continue;
            }
            if (token[0] == '#') {
                return;
            }
            if (processQuotedText(token))
                continue;
            if (processKeyValueArg(token))
                continue;
            if (processSeconds(token))
                continue;
            if (processUserGuid(token))
                continue;
            if (processCallStatus(token))
                continue;
            if (processName(token))
                continue;
            addFreeText(token);
        }
    }

    void add(const CommandData& other) {
        Y_VERIFY(other.name.empty());
        args.insert(std::begin(other.args), std::end(other.args));
        usedArgs.insert(std::begin(other.usedArgs), std::end(other.usedArgs));
        if (!other.freeText.empty())
            freeText += " " + other.freeText;
    }

    bool processKeyValueArg(const std::string& token) {
        size_t m;
        if ((m = token.find(":")) != std::string::npos) {
            auto arg = token.substr(0, m);
            if (args.find(arg) != args.end()) {
                throw std::runtime_error("Duplicated arguments for command " +
                                         toString());
            }
            auto val = token.substr(m + 1);
            args[arg] = val;
            return true;
        }
        return false;
    }

    bool processSeconds(const std::string& token) {
        if (hasEnding(token, "s")) {
            auto it = token.begin();
            while (it != (token.end() - 1) && std::isdigit(*it))
                ++it;
            if (it == token.end() - 1) {
                args["seconds"] = token.substr(0, token.size() - 1);
                return true;
            }
        }
        if (hasEnding(token, "sec")) {
            auto it = token.begin();
            while (it != (token.end() - 3) && std::isdigit(*it))
                ++it;
            if (it == token.end() - 3) {
                args["seconds"] = token.substr(0, token.size() - 3);
                return true;
            }
        }
        return false;
    }

    bool processCallStatus(const std::string& token) {
        try {
            fromString(token);
        } catch (std::runtime_error e) {
            return false;
        }
        std::string val = token;
        args["state"] = val;
        return true;
    }

    bool hasEnding(const std::string& text, const std::string& ending) {
        if (text.length() >= ending.length()) {
            return (0 == text.compare(text.length() - ending.length(),
                                      ending.length(), ending));
        } else {
            return false;
        }
    }

    bool processName(const std::string& token) {
        if (name.empty()) {
            name = token;
            return true;
        }
        return false;
    }

    bool processUserGuid(const std::string& token) {
        if (quasar::isUUID(token)) {
            args["guid"] = token;
            return true;
        }
        return false;
    }

    bool processQuotedText(const std::string& token) {
        if (token.size() == 1) {
            if (token.front() != '"') {
                if (collectingQuotedText) {
                    addFreeText("");
                    return false;
                }
                return false;
            }
            addFreeText("");
            collectingQuotedText = !collectingQuotedText;
            return true;
        }
        if (token.front() == '"' && token.back() == '"') {
            if (token.size() > 2) {
                addFreeText(token.substr(1, token.size() - 2));
            }
            return true;
        }
        if (!collectingQuotedText) {
            if (token.front() == '"') {
                collectingQuotedText = true;
                addFreeText(token.substr(1));
                return true;
            } else {
                return false;
            }
        } else {
            if (token.back() == '"') {
                addFreeText(token.substr(0, token.size() - 1));
                collectingQuotedText = false;
                return true;
            } else {
                addFreeText(token);
                return true;
            }
        }
    }

    void addFreeText(const std::string& word) {
        if (!freeText.empty())
            freeText += " ";
        freeText += word;
    }

    bool empty() {
        return args.empty() && name.empty() && freeText.empty();
    }

    std::string toString() {
        std::string output = name;
        for (auto& entry : args) {
            output += " " + (entry.first) + ":" + (entry.second);
        }
        for (auto& entry : usedArgs) {
            output += " " + (entry.first) + ":" + (entry.second);
        }
        if (!freeText.empty()) {
            output += " " + freeText;
        }
        return output;
    }

    Value get(const std::string& arg) {
        auto it = args.find(arg);
        if (it == args.end()) {
            throw std::runtime_error("Argument '" + arg +
                                     "' is required for command '" + name +
                                     "'");
        }
        auto value = it->second;
        usedArgs[arg] = value;
        args.erase(it);
        return value;
    }

    Value get(const std::string& arg, const Value& def) {
        auto it = args.find(arg);
        auto value = (it == args.end()) ? (std::string)def : it->second;
        usedArgs[arg] = value;
        if (it != args.end())
            args.erase(it);
        return value;
    }

    void checkUnusedArgs() {
        if (args.empty())
            return;
        std::string output;
        for (auto& entry : args) {
            if (!output.empty())
                output += ", ";
            output += entry.first;
        }
        throw std::runtime_error("There are invalid arguments for command: " +
                                 output);
    }

    std::string name;
    bool collectingQuotedText = false;
    std::string freeText;
    std::map<std::string, std::string> args;
    std::map<std::string, std::string> usedArgs;
    size_t lineNumber;
};

DirectToolProcessor::DirectToolProcessor(std::shared_ptr<Printer> printer, std::shared_ptr<Session> session)
    : ToolProcessor(printer)
    , thread_(LoopThread::create())
    , session_(std::move(session))
    , destroying_(false)
{
    aliases_["start"] = "startSession";
    aliases_["on"] = "startSession";
    aliases_["1"] = "startSession";
    aliases_["online"] = "sendHeartbeat";
    aliases_["o"] = "sendHeartbeat";
    aliases_["call"] = "startCall";
    aliases_["c"] = "startCall";
    aliases_["waitCall"] = "waitAnyCallRinging";
    aliases_["wait"] = "waitAnyCallRinging";
    aliases_["w"] = "waitAnyCallRinging";
    aliases_["waitIncoming"] = "waitIncomingCall";
    aliases_["wi"] = "waitIncomingCall";
    aliases_["waitOutgoing"] = "waitOutgoingCall";
    aliases_["wo"] = "waitOutgoingCall";
    aliases_["pause"] = "sleep";
    aliases_["p"] = "sleep";
    aliases_["decline"] = "declineIncomingCall";
    aliases_["d"] = "declineIncomingCall";
    aliases_["-"] = "declineIncomingCall";
    aliases_["accept"] = "acceptIncomingCall";
    aliases_["a"] = "acceptIncomingCall";
    aliases_["+"] = "acceptIncomingCall";
    aliases_["hangup"] = "hangupCall";
    aliases_["hang"] = "hangupCall";
    aliases_["stop"] = "hangupCall";
    aliases_["bye"] = "hangupCall";
    aliases_["h"] = "hangupCall";
    aliases_["."] = "hangupCall";
    aliases_["off"] = "finishSession";
    aliases_["0"] = "finishSession";
    aliases_["quit"] = "exitScript";
    aliases_["exit"] = "exitScript";
    aliases_["q"] = "exitScript";
    aliases_["@"] = "echo";

    watchNewSession();

    watchStateChange();
}

DirectToolProcessor::~DirectToolProcessor() {
    destroying_ = true;
    thread_->destroyBlocked();
}

void DirectToolProcessor::execute(const std::string& text) {
    size_t lineNumber = 1;
    try {
        std::stringstream stream(text);
        std::string line;
        std::unique_ptr<CommandData> currentCommand;
        while (std::getline(stream, line, '\n')) {
            std::unique_ptr<CommandData> data = std::make_unique<CommandData>(
                line, lineNumber++);
            if (data->empty())
                continue;
            if (data->name.empty()) {
                if (!currentCommand) {
                    throw std::runtime_error("Args without a command found");
                }
                currentCommand->add(*data);
            } else {
                if (currentCommand) {
                    execute(*currentCommand);
                }
                currentCommand = std::move(data);
            }
        }
        if (currentCommand) {
            execute(*currentCommand);
        }
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("DirectToolProcessor.ExecuteTextCommand.Exception", e.what());
        YIO_LOG_DEBUG(">>> Command dump begin\n"
                      << text << "\n<<< Command dump end");
        printer_->printerrln(e.what());
    } catch (ExitScriptSignal s) {
        YIO_LOG_DEBUG(">>> Commands break on line " << lineNumber);
    }
}

void DirectToolProcessor::processCommandName(CommandData& command) {
    auto& name = command.name;
    if (name.empty()) {
        return;
    }
    auto it = aliases_.find(name);
    if (it != std::end(aliases_)) {
        name = it->second;
    }
}

void DirectToolProcessor::execute(CommandData& command) {
    processCommandName(command);
    const auto& name = command.name;
    YIO_LOG_DEBUG("COMMAND starting: " << name << " (line " << command.lineNumber
                                       << ")");
    try {
        if (name == "sendHeartbeat") {
            sendHeartbeat(command);
        } else if (name == "startCall") {
            startCall(command);
        } else if (name == "waitAnyCallRinging") {
            waitAnyCallRinging(command);
        } else if (name == "waitIncomingCall") {
            waitIncomingCall(command);
        } else if (name == "waitOutgoingCall") {
            waitOutgoingCall(command);
        } else if (name == "sleep") {
            sleep(command);
        } else if (name == "declineIncomingCall") {
            declineIncomingCall(command);
        } else if (name == "acceptIncomingCall") {
            acceptIncomingCall(command);
        } else if (name == "hangupCall") {
            hangupCall(command);
        } else if (name == "exitScript") {
            exitScript(command);
        } else if (name == "echo") {
            echo(command);
        } else {
            throw std::runtime_error("Unknown command " + name);
        }
        YIO_LOG_DEBUG("COMMAND finished: " << command.name);
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("DirectToolProcessor.ExecuteCommand.Exception", "COMMAND failed: " << command.name);
        throw std::runtime_error(std::string("  Command failed (line ") +
                                 std::to_string(command.lineNumber) + "): " +
                                 e.what() + "\n  " + command.toString());
    }
    processStateChange();
}

void DirectToolProcessor::processStateChange() {
    std::lock_guard<std::mutex> lock(sessionMutex_);

    std::string currentStateString = "> State: ";

    auto state = session_->getState();

    if (!state.authorized) {
        currentStateString += "not authorized";
    } else if (!state.connected) {
        currentStateString += "not connected";
    } else if (state.status == Status::NOCALL) {
        currentStateString += "connected, no call";
    } else if (state.direction == rtc::Direction::INCOMING) {
        currentStateString += "incoming call, ";
        currentStateString += toString(state.status);
        currentStateString += ", from '" + state.userName + "' (" + state.userGuid + ") avatar: " + state.userAvatar;
    } else if (state.direction == rtc::Direction::OUTGOING) {
        currentStateString += "outgoing call, ";
        currentStateString += toString(state.status);
        currentStateString += ", to '" + state.userName + "' (" + state.userGuid + ") avatar: " + state.userAvatar;
    }
    if (currentStateString != lastStateString_) {
        lastStateString_ = currentStateString;
        printer_->printinfoln(currentStateString);
    }
}

void DirectToolProcessor::watchStateChange() {
    processStateChange();
    // destroyBlocked() makes capturing this safe.
    thread_->executeDelayed([this] { watchStateChange(); },
                            std::chrono::milliseconds(500));
}

void DirectToolProcessor::watchNewSession() {
    processStateChange();
    callStateSubscription_ = session_->subscribeStateChanged([this](const Session::State&) {
        // destroyBlocked() makes capturing this safe.
        thread_->execute([this] { processStateChange(); });
    });
    callCreationFailedSubscription_ = session_->subscribeCallCreationFailed([this] {
        // destroyBlocked() makes capturing this safe.
        thread_->execute([this] {
            std::lock_guard<std::mutex> lock(sessionMutex_);
            printer_->printerrln("Call creation failed");
        });
    });
}

void DirectToolProcessor::checkConnected() {
    if (destroying_)
        return;
    auto state = session_->getState();
    if (!state.authorized || !state.connected) {
        throw std::runtime_error("Connection lost");
    }
}

void DirectToolProcessor::sleepSeconds(unsigned seconds) {
    std::this_thread::sleep_for(std::chrono::seconds(seconds));
    checkConnected();
}

void DirectToolProcessor::exitScript(CommandData& command) {
    command.checkUnusedArgs();
    if (!command.freeText.empty()) {
        printer_->printwarnln("@ " + command.freeText);
    }
    throw ExitScriptSignal();
}

void DirectToolProcessor::echo(CommandData& command) {
    command.checkUnusedArgs();
    printer_->printwarnln("@ " + command.freeText);
}

void DirectToolProcessor::sleep(CommandData& command) {
    unsigned seconds = command.get("seconds");
    command.checkUnusedArgs();

    sleepSeconds(seconds);
}

bool DirectToolProcessor::hasIncomingCall(const Session::State& state, Status passedStatus) {
    if (state.status != Status::NOCALL && state.direction == rtc::Direction::INCOMING) {
        return !(state.status < passedStatus);
    }

    return false;
}

bool DirectToolProcessor::hasOutgoingCall(const Session::State& state, Status passedStatus) {
    if (state.status != Status::NOCALL && state.direction == rtc::Direction::OUTGOING) {
        return !(state.status < passedStatus);
    }

    return false;
}

bool DirectToolProcessor::hasIncomingCall(Status passedStatus) {
    auto state = session_->getState();
    if (passedStatus == Status::ENDED && !hasIncomingCall(state)) {
        // ENDED immediately leads to 'no call'
        return true;
    }
    return hasIncomingCall(state, passedStatus);
}

bool DirectToolProcessor::hasOutgoingCall(Status passedStatus) {
    auto state = session_->getState();
    if (passedStatus == Status::ENDED && !hasOutgoingCall(state)) {
        // ENDED can lead to 'no call'
        return true;
    }
    return hasOutgoingCall(state, passedStatus);
}

void DirectToolProcessor::outputWaiting(unsigned seconds) {
    if (seconds == 0) {
        throw std::runtime_error("timeout");
    }
    if (seconds % 10 == 0) {
        printer_->println("Waiting " + std::to_string(seconds) + " sec...");
    }
}

void DirectToolProcessor::sendHeartbeat(CommandData& /* command */) {
    session_->sendHeartbeat();
    checkConnected();
}

void DirectToolProcessor::startCall(CommandData& command) {
    std::string toGuid = command.get("guid");
    command.checkUnusedArgs();

    session_->startCall(toGuid, "");
    checkConnected();
}

void DirectToolProcessor::waitAnyCallRinging(CommandData& command) {
    unsigned seconds = command.get("seconds", 30);
    std::string forState = command.get("state", "RINGING");
    command.checkUnusedArgs();

    auto state = fromString(forState);
    if (state == Status::ENDED) {
        auto ended = [&] {
            auto sessionState = session_->getState();

            bool noCall = !hasIncomingCall(sessionState) &&
                          !hasOutgoingCall(sessionState);
            bool callEnded = hasIncomingCall(sessionState, Status::ENDED) ||
                             hasOutgoingCall(sessionState, Status::ENDED);
            return noCall || callEnded;
        };
        while (!ended()) {
            if (destroying_)
                return;
            outputWaiting(seconds);
            sleepSeconds(1);
            seconds--;
        }
    } else {
        while (!hasIncomingCall(state) && !hasOutgoingCall(state)) {
            if (destroying_)
                return;
            outputWaiting(seconds);
            sleepSeconds(1);
            seconds--;
        }
    }
    checkConnected();
}

void DirectToolProcessor::waitIncomingCall(CommandData& command) {
    unsigned seconds = command.get("seconds", 30);
    std::string forState = command.get("state", "RINGING");
    command.checkUnusedArgs();

    auto state = fromString(forState);
    while (!hasIncomingCall(state)) {
        if (destroying_)
            return;
        outputWaiting(seconds);
        sleepSeconds(1);
        seconds--;
    }
    checkConnected();
}

void DirectToolProcessor::waitOutgoingCall(CommandData& command) {
    unsigned seconds = command.get("seconds", 30);
    std::string forState = command.get("state", "DIALING");
    command.checkUnusedArgs();

    auto state = fromString(forState);
    while (!hasOutgoingCall(state)) {
        if (destroying_)
            return;
        outputWaiting(seconds);
        sleepSeconds(1);
        seconds--;
    }
    checkConnected();
}

void DirectToolProcessor::declineIncomingCall(CommandData& command) {
    command.checkUnusedArgs();

    auto state = session_->getState();
    if (!hasIncomingCall(state, Status::RINGING)) {
        throw std::runtime_error("No incoming call to decline");
    }
    session_->declineIncomingCall();
    checkConnected();
}

void DirectToolProcessor::acceptIncomingCall(CommandData& command) {
    command.checkUnusedArgs();

    auto state = session_->getState();
    if (!hasIncomingCall(state, Status::RINGING)) {
        throw std::runtime_error("No incoming call to accept");
    }
    session_->acceptIncomingCall();
    checkConnected();
}

void DirectToolProcessor::hangupCall(CommandData& command) {
    command.checkUnusedArgs();

    auto state = session_->getState();
    if (!hasIncomingCall(state, Status::DIALING) &&
        !hasOutgoingCall(state, Status::DIALING)) {
        throw std::runtime_error("No call to hangup");
    }
    session_->hangupCall();
    checkConnected();
}
