#include "mediator_api.h"

#include <yandex_io/libs/base/utils.h>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger::rtc;

namespace {

    std::string errorTypeToString(messenger::rtc::MediatorApi::ErrorType type) {
        switch (type) {
            case messenger::rtc::MediatorApi::ErrorType::INVALID_OFFER:
                return "invalidOffer";

            case messenger::rtc::MediatorApi::ErrorType::INVALID_ANSWER:
                return "invalidAnswer";

            case messenger::rtc::MediatorApi::ErrorType::ADD_CANDIDATE_FAILED:
                return "invalidIceCandidates";

            case messenger::rtc::MediatorApi::ErrorType::REMOVE_CANDIDATES_FAILED:
                return "removeCandidatesError";

            case messenger::rtc::MediatorApi::ErrorType::CLIENT_ERROR:
                return "clientError";

            default:
                Y_VERIFY(false);

                return "";
        }
    }

    enum class ErrorCode: int {
        ERROR_CODE_DEFAULT = -32050,
        ERROR_CODE_INVALID_ICE_CANDIDATES = -32053,
    };

    ErrorCode errorTypeToCode(messenger::rtc::MediatorApi::ErrorType type) {
        switch (type) {
            case messenger::rtc::MediatorApi::ErrorType::ADD_CANDIDATE_FAILED:
                return ErrorCode::ERROR_CODE_INVALID_ICE_CANDIDATES;

            default:
                return ErrorCode::ERROR_CODE_DEFAULT;
        }
    }

} // namespace

MediatorApi::MediatorApi(std::shared_ptr<Transport> transport,
                         std::shared_ptr<LoopThread> workerThread)
    : transport_(std::move(transport))
    , workerThread_(std::move(workerThread))
{
}

std::shared_ptr<MediatorApi>
MediatorApi::create(std::shared_ptr<Transport> transport,
                    std::shared_ptr<LoopThread> workerThread) {
    Y_VERIFY(workerThread->checkInside());

    auto api = std::make_shared<MediatorApi>(std::move(transport),
                                             std::move(workerThread));

    api->transport_->setListener(api);

    return api;
}

void MediatorApi::onMessage(const Transport::Message& msg) {
    Y_VERIFY(workerThread_->checkInside());

    MediatorApi::Message message;

    try {
        message = quasar::parseJson(msg);

    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("MediatorApi.OnMessage.InvalidJson", "Can't parse mediator message: " << e.what());
        return;
    }

    observerList_.notifyObservers(message);
}

MediatorApi::MessageSubscription
MediatorApi::subscribe(MediatorApi::MessageObserver observer) {
    return observerList_.subscribe(std::move(observer));
}

void MediatorApi::send(const MediatorRequest& request) {
    Y_VERIFY(workerThread_->checkInside());

    transport_->send(quasar::jsonToString(request));
}

MediatorApi::MediatorRequest
MediatorApi::makeRequest(const std::string& id, const std::string& method) {
    MediatorRequest request;

    request["id"] = id;
    request["method"] = method;
    request["jsonrpc"] = "2.0";
    request["params"];

    return request;
}

std::string MediatorApi::retrieveIceServersConfig(const std::string& guid) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "getIceServersConfig");
    request["params"]["guid"] = guid;

    send(request);

    return transactionId;
}

std::string MediatorApi::offer(const std::string& guid,
                               const std::string& offer) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "offer");
    request["params"]["guid"] = guid;
    request["params"]["offer"] = offer;

    send(request);

    return transactionId;
}

std::string MediatorApi::answer(const std::string& guid,
                                const std::string& answer) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "answer");
    request["params"]["guid"] = guid;
    request["params"]["answer"] = answer;

    send(request);

    return transactionId;
}

std::string
MediatorApi::addCandidates(const std::string& guid,
                           const std::vector<IceCandidate>& candidates) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "addCandidates");

    request["params"]["guid"] = guid;

    Json::Value candidatesJson(Json::arrayValue);
    for (const IceCandidate& candidate : candidates) {
        Json::Value candidateJson;
        candidateJson["m_id"] = candidate.mId;
        candidateJson["m_line_index"] = candidate.mLineIndex;
        candidateJson["candidate"] = candidate.candidate;
        candidatesJson.append(std::move(candidateJson));
    }

    request["params"]["candidates"] = candidatesJson;

    send(request);

    return transactionId;
}

std::string
MediatorApi::removeCandidates(const std::string& guid,
                              const std::vector<IceCandidate>& candidates) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "removeCandidates");

    request["params"]["guid"] = guid;

    Json::Value candidatesJson(Json::arrayValue);
    for (const IceCandidate& candidate : candidates) {
        Json::Value candidateJson;
        candidateJson["m_id"] = candidate.mId;
        candidateJson["m_line_index"] = candidate.mLineIndex;
        candidateJson["candidate"] = candidate.candidate;
        candidatesJson.append(std::move(candidateJson));
    }

    request["params"]["candidates"] = candidatesJson;

    send(request);

    return transactionId;
}

std::string MediatorApi::acknowledge(const std::string& guid,
                                     const std::string& requestId) {
    Json::Value ack;
    ack["id"] = requestId;
    ack["result"]["guid"] = guid;
    ack["jsonrpc"] = "2.0";

    send(ack);

    return requestId;
}

std::string MediatorApi::updateMediaState(const std::string& guid,
                                          bool sendAudio, bool sendVideo) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "updateState");
    request["params"]["guid"] = guid;
    request["params"]["media_state"]["audio"] = sendAudio;
    request["params"]["media_state"]["video"] = sendVideo;

    send(request);

    return transactionId;
}

std::string MediatorApi::keepAlive(const std::string& guid,
                                   const Json::Value& statsReport) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "keepAlive");
    request["params"]["guid"] = guid;
    request["params"]["stats"] = statsReport;

    send(request);

    return transactionId;
}

std::string MediatorApi::reportError(const std::string& guid,
                                     const std::string& message,
                                     ErrorType type) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "error");
    request["params"]["guid"] = guid;
    request["params"]["message"] = message;
    request["params"]["type"] = errorTypeToString(type);

    send(request);

    return transactionId;
}

std::string MediatorApi::reportEvent(const std::string& guid,
                                     const Json::Value& event) {
    std::string transactionId = quasar::makeUUID();

    MediatorRequest request = makeRequest(transactionId, "reportEvent");
    request["params"]["guid"] = guid;
    request["params"]["event"] = event;

    send(request);

    return transactionId;
}

void MediatorApi::returnError(const std::string& guid,
                              const std::string& requestId,
                              const std::string& message,
                              const Json::Value& detail, ErrorType type) {
    Json::Value error;

    error["jsonrpc"] = "2.0";
    error["requestId"] = requestId;

    error["error"]["code"] = static_cast<int>(errorTypeToCode(type));
    error["error"]["message"] = message;

    error["error"]["data"]["guid"] = guid;
    error["error"]["data"]["type"] = errorTypeToString(type);
    error["error"]["data"]["detail"] = detail;

    send(error);
}
