#include "redirect_peer_connection_observer.h"

#include "entities.h"

#include <vector>

#include <util/system/yassert.h>

using namespace messenger::rtc;

RedirectPeerConnectionObserver::RedirectPeerConnectionObserver(
    std::shared_ptr<LoopThread> workerThread)
    : workerThread_(workerThread)
{
    ensureOnWorkerThread();
}

RedirectPeerConnectionObserver::~RedirectPeerConnectionObserver() {
    ensureOnWorkerThread();
}

std::weak_ptr<RedirectPeerConnectionObserver> RedirectPeerConnectionObserver::weakFromThis() {
    return shared_from_this();
}

void RedirectPeerConnectionObserver::ensureOnWorkerThread() {
    Y_VERIFY(workerThread_->checkInside());
}

void RedirectPeerConnectionObserver::subscribe(
    webrtc::PeerConnectionObserver* observer) {
    ensureOnWorkerThread();

    observers_.insert(observer);
}

void RedirectPeerConnectionObserver::unsubscribe(webrtc::PeerConnectionObserver* observer) {
    ensureOnWorkerThread();

    observers_.erase(observer);
}

void RedirectPeerConnectionObserver::OnSignalingChange(
    webrtc::PeerConnectionInterface::SignalingState newState) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([newState, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnSignalingChange(newState);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnIceConnectionChange(
    webrtc::PeerConnectionInterface::IceConnectionState newState) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([newState, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnIceConnectionChange(newState);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnConnectionChange(
    webrtc::PeerConnectionInterface::PeerConnectionState newState) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([newState, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnConnectionChange(newState);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnIceConnectionReceivingChange(
    bool receiving) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([receiving, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnIceConnectionReceivingChange(receiving);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnIceGatheringChange(
    webrtc::PeerConnectionInterface::IceGatheringState newState) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([newState, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnIceGatheringChange(newState);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnIceCandidate(const webrtc::IceCandidateInterface* candidate) {
    IceCandidate c;
    c.mId = candidate->sdp_mid();
    c.mLineIndex = candidate->sdp_mline_index();
    if (!candidate->ToString(&c.candidate)) {
        throw std::runtime_error("Unexpected candidate");
    }

    const auto weakRef = weakFromThis();

    workerThread_->execute([c, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            webrtc::SdpParseError error;
            std::unique_ptr<webrtc::IceCandidateInterface> cc(
                webrtc::CreateIceCandidate(c.mId, c.mLineIndex, c.candidate,
                                           &error));
            if (!cc) {
                throw std::runtime_error("Unexpected candidate");
            }

            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnIceCandidate(cc.get());
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnIceCandidatesRemoved(
    const std::vector<cricket::Candidate>& candidates) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([candidates, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnIceCandidatesRemoved(candidates);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnAddStream(
    ::rtc::scoped_refptr<webrtc::MediaStreamInterface> stream) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([stream, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnAddStream(stream);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnRemoveStream(
    ::rtc::scoped_refptr<webrtc::MediaStreamInterface> stream) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([stream, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnRemoveStream(stream);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnDataChannel(
    ::rtc::scoped_refptr<webrtc::DataChannelInterface> dataChannel) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([dataChannel, this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnDataChannel(dataChannel);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnRenegotiationNeeded() {
    const auto weakRef = weakFromThis();

    workerThread_->execute([this, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnRenegotiationNeeded();
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnAddTrack(
    ::rtc::scoped_refptr<webrtc::RtpReceiverInterface> receiver,
    const std::vector<::rtc::scoped_refptr<webrtc::MediaStreamInterface>>&
        streams) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([this, receiver, streams, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnAddTrack(receiver, streams);
                }
            }
        }
    });
}

void RedirectPeerConnectionObserver::OnTrack(
    ::rtc::scoped_refptr<webrtc::RtpTransceiverInterface> transceiver) {
    const auto weakRef = weakFromThis();

    workerThread_->execute([this, transceiver, weakRef]() {
        if (const auto ref = weakRef.lock()) {
            // Copy observers_ because it can be modified in callbacks
            std::unordered_set<webrtc::PeerConnectionObserver*> copy(
                observers_.cbegin(), observers_.cend());

            for (webrtc::PeerConnectionObserver* o : copy) {
                if (observers_.count(o)) {
                    o->OnTrack(transceiver);
                }
            }
        }
    });
}
