#include "sender_state_machine.h"

#include "states/complete_sending_state.h"
#include "states/waiting_for_candidates_state.h"

#include <yandex_io/callkit/util/loop_thread.h>

#include <util/system/yassert.h>

#include <memory>
#include <unordered_set>

using namespace messenger::rtc;

SenderStateMachine::SenderStateMachine(const std::string& sessionUuid,
                                       std::shared_ptr<MediatorApi> mediator,
                                       std::shared_ptr<LoopThread> workerThread)
    : sessionUuid_(sessionUuid)
    , mediator_(mediator)
    , workerThread_(workerThread)
{
    ensureOnWorkerThread();

    setState(std::make_unique<InitialState>());
}

SenderStateMachine::~SenderStateMachine() {
    ensureOnWorkerThread();

    setState(nullptr);
}

void SenderStateMachine::reset() {
    ensureOnWorkerThread();

    unsentCandidates_.clear();
    gatheringCompleted_ = false;
    setState(std::make_unique<InitialState>());
}

void SenderStateMachine::start() {
    ensureOnWorkerThread();

    if (getState()->name() == InitialState::NAME) {
        setState(std::make_unique<WaitingForCandidatesState>(this));
    }
}

void SenderStateMachine::addCandidate(const IceCandidate& candidate) {
    ensureOnWorkerThread();

    unsentCandidates_.push_back(std::make_pair(candidate, Operation::ADD));

    trySend();
}

void SenderStateMachine::removeCandidates(const std::vector<IceCandidate>& candidates) {
    ensureOnWorkerThread();

    for (const auto& c : candidates) {
        unsentCandidates_.push_back(std::make_pair(c, Operation::REMOVE));
    }

    trySend();
}

void SenderStateMachine::completeTrickling() {
    ensureOnWorkerThread();

    gatheringCompleted_ = true;

    tryComplete();
}

void SenderStateMachine::trySend() {
    if (getState()->name() == WaitingForCandidatesState::NAME) {
        setState(std::make_unique<WaitingForCandidatesState>(this));
    }
}

void SenderStateMachine::tryComplete() {
    if (getState()->name() == WaitingForCandidatesState::NAME) {
        setState(std::make_unique<CompleteSendingState>(this));
    }
}

void SenderStateMachine::ensureOnWorkerThread() const {
    Y_VERIFY(workerThread_->checkInside());
}

void SenderStateMachine::removeSentCandidates(
    Operation operation, const std::vector<IceCandidate>& candidates) {
    const std::unordered_set<IceCandidate, IceCandidate::Hash> candidatesSet(
        candidates.cbegin(), candidates.cend());

    auto it = unsentCandidates_.begin();

    while (it != unsentCandidates_.end()) {
        if (it->second == operation && candidatesSet.count(it->first) != 0) {
            it = unsentCandidates_.erase(it);

        } else {
            ++it;
        }
    }
}

void SenderStateMachine::onStateChange(State* oldState, State* newState) {
    ensureOnWorkerThread();

    YIO_LOG_INFO("SenderStateMachine: " << (oldState ? oldState->name() : "N/A") << "->" << (newState ? newState->name() : "N/A"));
}

SenderStateMachine::SenderState::SenderState(SenderStateMachine* machine)
    : machine_(machine)
{
    machine_->ensureOnWorkerThread();
}
