#include "session_state_machine.h"

#include "states/config_awaiting_state.h"

#include <yandex_io/callkit/util/loop_thread.h>

#include <contrib/libs/webrtc/api/create_peerconnection_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/audio_decoder_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/audio_encoder_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/builtin_audio_decoder_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/builtin_audio_encoder_factory.h>
#include <contrib/libs/webrtc/modules/audio_processing/include/audio_processing.h>

#include <util/system/yassert.h>

#include <memory>

using namespace messenger::rtc;

SessionStateMachine::SessionState::SessionState(SessionStateMachine* machine)
    : machine_(machine)
{
    machine_->ensureOnWorkerThread();
}

SessionStateMachine::SessionStateMachine(
    const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory,
    Direction direction,
    const std::string& sessionUuid,
    std::shared_ptr<MediatorApi> mediator,
    std::shared_ptr<LoopThread> workerThread,
    std::shared_ptr<TimerFactory> timerFactory,
    std::shared_ptr<quasar::IUserConfigProvider> userConfigProvider,
    Json::Value callsConfig)
    : ipcFactory_(ipcFactory)
    , direction_(direction)
    , sessionUuid_(sessionUuid)
    , mediator_(mediator)
    , workerThread_(workerThread)
    , timerFactory_(timerFactory)
    , status_(MediaSession::Status::DISCONNECTED)
    , listener_(nullptr)
    , userConfigProvider_(std::move(userConfigProvider))
    , connectionObserver_(
          std::make_shared<RedirectPeerConnectionObserver>(workerThread_))
    , callsConfig_(std::move(callsConfig))
    , audio_(quasar::QuasarAudioDeviceModule::create(ipcFactory_, userConfigProvider_, callsConfig_))
    , apm_(webrtc::AudioProcessingBuilder().Create())
    , webrtcSignalingThread_(::rtc::Thread::Create().release())
    , webrtcWorkerThread_(::rtc::Thread::Create().release())
{
    ensureOnWorkerThread();
    setState(std::make_unique<ConfigAwaitingState>(this));

    apm_->ApplyConfig(audio_->getAudioProcessingConfig());

    webrtcSignalingThread_->Start();
    webrtcWorkerThread_->Start();
}

SessionStateMachine::~SessionStateMachine() {
    ensureOnWorkerThread();

    setState(nullptr);
}

void SessionStateMachine::setListener(SessionStateMachine::Listener* listener) {
    ensureOnWorkerThread();

    listener_ = listener;
}

MediaSession::Status SessionStateMachine::getStatus() const {
    ensureOnWorkerThread();

    return status_;
}

void SessionStateMachine::onStateChange(State* oldState, State* newState) {
    ensureOnWorkerThread();

    YIO_LOG_INFO("SessionStateMachine: " << (oldState ? oldState->name() : "N/A") << "->" << (newState ? newState->name() : "N/A"));
}

void SessionStateMachine::ensureOnWorkerThread() const {
    Y_VERIFY(workerThread_->checkInside());
}

void SessionStateMachine::subscribe(webrtc::PeerConnectionObserver* observer) {
    connectionObserver_->subscribe(observer);
}

void SessionStateMachine::unsubscribe(webrtc::PeerConnectionObserver* observer) {
    connectionObserver_->unsubscribe(observer);
}

void SessionStateMachine::notifyStatusChange(MediaSession::Status status) {
    ensureOnWorkerThread();

    if (listener_) {
        listener_->onStatusChanged(status);
    }
}

void SessionStateMachine::notifyFailure(const std::string& errorMessage) {
    ensureOnWorkerThread();

    if (listener_) {
        listener_->onFailure(errorMessage);
    }
}

void SessionStateMachine::resetAudio() {
    ensureOnWorkerThread();

    audio_ = quasar::QuasarAudioDeviceModule::create(ipcFactory_, userConfigProvider_, callsConfig_);
}
