#include "config_applying_state.h"

#include "peer_connection_creating_state.h"
#include "peer_connection_creation_failed_state.h"

#include <yandex_io/callkit/rtc/media/fake_webrtc_video_engine/fake_web_rtc_video_engine.h>

#include <contrib/libs/webrtc/api/create_peerconnection_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/audio_decoder_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/audio_encoder_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/builtin_audio_decoder_factory.h>
#include <contrib/libs/webrtc/api/audio_codecs/builtin_audio_encoder_factory.h>

#include <memory>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger::rtc;

const std::string SessionStateMachine::ConfigApplyingState::NAME = "ConfigApplyingState";

SessionStateMachine::ConfigApplyingState::ConfigApplyingState(SessionStateMachine* machine, const Json::Value& config)
    : SessionState(machine)
    , config_(config)
{
}

void SessionStateMachine::ConfigApplyingState::enter() {
    machine_->config_ = config_;

    webrtc::PeerConnectionInterface::RTCConfiguration rtcConfig;

    // Client defaults
    rtcConfig.sdp_semantics = webrtc::SdpSemantics::kUnifiedPlan;

    // Backend overrides
    const auto& cfg = config_["rtc_configuration"];

    if (cfg.isMember("ice_servers")) {
        for (const Json::Value& v : cfg["ice_servers"]) {
            webrtc::PeerConnectionInterface::IceServer iceServer;

            if (v.isMember("urls")) {
                for (const Json::Value& u : v["urls"]) {
                    iceServer.urls.push_back(u.asString());
                }
            }

            if (v.isMember("username")) {
                iceServer.username = v["username"].asString();
            }

            if (v.isMember("credential")) {
                iceServer.password = v["credential"].asString();
            }

            rtcConfig.servers.push_back(iceServer);
        }
    }

    if (cfg.isMember("ice_transport_policy")) {
        const std::string v = cfg["ice_transport_policy"].asString();

        if (v == "none") {
            rtcConfig.type = webrtc::PeerConnectionInterface::kNone;

        } else if (v == "relay") {
            rtcConfig.type = webrtc::PeerConnectionInterface::kRelay;

        } else if (v == "nohost") {
            rtcConfig.type = webrtc::PeerConnectionInterface::kNoHost;

        } else if (v == "all") {
            rtcConfig.type = webrtc::PeerConnectionInterface::kAll;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedIceTransportPolicy", "Unexpected ice_transport_policy value: " << v);
        }
    }

    if (cfg.isMember("gathering_policy")) {
        const std::string v = cfg["gathering_policy"].asString();

        if (v == "once") {
            rtcConfig.continual_gathering_policy = webrtc::PeerConnectionInterface::GATHER_ONCE;

        } else if (v == "continually") {
            rtcConfig.continual_gathering_policy = webrtc::PeerConnectionInterface::GATHER_CONTINUALLY;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedGatheringPolicy", "Unexpected gathering_policy value: " << v);
        }
    }

    if (cfg.isMember("ice_regather_interval_range")) {
        YIO_LOG_ERROR_EVENT("ConfigApplyingState.IceRegatherIntervalRangeIsNotSupported", "Can't set ice_regather_interval_range");
    }

    if (cfg.isMember("sdp_semantics")) {
        const std::string v = cfg["sdp_semantics"].asString();

        if (v == "unified-plan") {
            rtcConfig.sdp_semantics = webrtc::SdpSemantics::kUnifiedPlan;

        } else if (v == "plan-b") {
            rtcConfig.sdp_semantics = webrtc::SdpSemantics::kPlanB;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedSdpSemantics", "Unexpected sdp_semantics value: " << v);
        }
    }

    if (cfg.isMember("bundle_policy")) {
        const std::string v = cfg["bundle_policy"].asString();

        if (v == "balanced") {
            rtcConfig.bundle_policy = webrtc::PeerConnectionInterface::kBundlePolicyBalanced;

        } else if (v == "max-compat") {
            rtcConfig.bundle_policy = webrtc::PeerConnectionInterface::kBundlePolicyMaxCompat;

        } else if (v == "max-bundle") {
            rtcConfig.bundle_policy = webrtc::PeerConnectionInterface::kBundlePolicyMaxBundle;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedBundlePoliy", "Unexpected bundle_policy value: " << v);
        }
    }

    if (cfg.isMember("rtcp_mux_policy")) {
        const std::string v = cfg["rtcp_mux_policy"].asString();

        if (v == "negotiate") {
            rtcConfig.rtcp_mux_policy = webrtc::PeerConnectionInterface::kRtcpMuxPolicyNegotiate;

        } else if (v == "require") {
            rtcConfig.rtcp_mux_policy = webrtc::PeerConnectionInterface::kRtcpMuxPolicyRequire;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedRtcpMuxPolicy", "Unexpected rtcp_mux_policy value: " << v);
        }
    }

    if (cfg.isMember("ice_candidate_pool_size")) {
        rtcConfig.ice_candidate_pool_size = cfg["ice_candidate_pool_size"].asInt();
    }

    if (cfg.isMember("tcp_candidate_policy")) {
        const std::string v = cfg["tcp_candidate_policy"].asString();

        if (v == "enabled") {
            rtcConfig.tcp_candidate_policy = webrtc::PeerConnectionInterface::kTcpCandidatePolicyEnabled;

        } else if (v == "disabled") {
            rtcConfig.tcp_candidate_policy = webrtc::PeerConnectionInterface::kTcpCandidatePolicyDisabled;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedTcpCandidatePolicy", "Unexpected tcp_candidate_policy value: " << v);
        }
    }

    if (cfg.isMember("candidate_network_policy")) {
        const std::string v = cfg["candidate_network_policy"].asString();

        if (v == "all") {
            rtcConfig.candidate_network_policy = webrtc::PeerConnectionInterface::kCandidateNetworkPolicyAll;

        } else if (v == "low-cost") {
            rtcConfig.candidate_network_policy = webrtc::PeerConnectionInterface::kCandidateNetworkPolicyLowCost;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedCandidateNetworkPolicy", "Unexpected candidate_network_policy value: " << v);
        }
    }

    if (cfg.isMember("audio_jitter_buffer_max_packets")) {
        rtcConfig.audio_jitter_buffer_max_packets = cfg["audio_jitter_buffer_max_packets"].asInt();
    }

    if (cfg.isMember("audio_jitter_buffer_fast_accelerate")) {
        rtcConfig.audio_jitter_buffer_fast_accelerate = cfg["audio_jitter_buffer_fast_accelerate"].asBool();
    }

    if (cfg.isMember("ice_connection_receiving_timeout")) {
        rtcConfig.ice_connection_receiving_timeout = cfg["ice_connection_receiving_timeout"].asInt();
    }

    if (cfg.isMember("ice_backup_candidate_pair_ping_interval")) {
        rtcConfig.ice_backup_candidate_pair_ping_interval = cfg["ice_backup_candidate_pair_ping_interval"].asInt();
    }

    if (cfg.isMember("ice_check_interval_strong_connectivity")) {
        rtcConfig.ice_check_interval_strong_connectivity = cfg["ice_check_interval_strong_connectivity"].asInt();
    }

    if (cfg.isMember("ice_check_interval_weak_connectivity")) {
        rtcConfig.ice_check_interval_weak_connectivity = cfg["ice_check_interval_weak_connectivity"].asInt();
    }

    if (cfg.isMember("ice_check_min_interval")) {
        rtcConfig.ice_check_min_interval = cfg["ice_check_min_interval"].asInt();
    }

    if (cfg.isMember("ice_unwritable_timeout")) {
        rtcConfig.ice_unwritable_timeout = cfg["ice_unwritable_timeout"].asInt();
    }

    if (cfg.isMember("ice_unwritable_min_checks")) {
        rtcConfig.ice_unwritable_min_checks = cfg["ice_unwritable_min_checks"].asInt();
    }

    if (cfg.isMember("prune_turn_ports")) {
        rtcConfig.prune_turn_ports = cfg["prune_turn_ports"].asBool();
    }

    if (cfg.isMember("presume_writable_when_fully_relayed")) {
        rtcConfig.presume_writable_when_fully_relayed = cfg["presume_writable_when_fully_relayed"].asBool();
    }

    if (cfg.isMember("disable_ipv6")) {
        rtcConfig.disable_ipv6 = cfg["disable_ipv6"].asBool();
    }

    if (cfg.isMember("disable_ipv6_on_wifi")) {
        rtcConfig.disable_ipv6_on_wifi = cfg["disable_ipv6_on_wifi"].asBool();
    }

    if (cfg.isMember("max_ipv6_networks")) {
        rtcConfig.max_ipv6_networks = cfg["max_ipv6_networks"].asInt();
    }

    if (cfg.isMember("enable_dscp")) {
        rtcConfig.set_dscp(cfg["enable_dscp"].asBool());
    }

    if (cfg.isMember("enable_cpu_overuse_detection")) {
        YIO_LOG_ERROR_EVENT("ConfigApplyingState.EnableCpuOveruseDetectionIsNotSupported", "Can't set enable_cpu_overuse_detection");
    }

    if (cfg.isMember("suspend_below_min_bitrate")) {
        rtcConfig.set_suspend_below_min_bitrate(cfg["suspend_below_min_bitrate"].asBool());
    }

    if (cfg.isMember("combined_audio_video_bwe")) {
        rtcConfig.combined_audio_video_bwe = cfg["combined_audio_video_bwe"].asBool();
    }

    if (cfg.isMember("network_preference")) {
        const std::string v = cfg["network_preference"].asString();

        if (v == "ethernet") {
            rtcConfig.network_preference = ::rtc::ADAPTER_TYPE_ETHERNET;

        } else if (v == "wifi") {
            rtcConfig.network_preference = ::rtc::ADAPTER_TYPE_WIFI;

        } else if (v == "cellular") {
            rtcConfig.network_preference = ::rtc::ADAPTER_TYPE_CELLULAR;

        } else if (v == "vpn") {
            rtcConfig.network_preference = ::rtc::ADAPTER_TYPE_VPN;

        } else if (v == "loopback") {
            rtcConfig.network_preference = ::rtc::ADAPTER_TYPE_LOOPBACK;

        } else if (v == "any") {
            rtcConfig.network_preference = ::rtc::ADAPTER_TYPE_ANY;

        } else {
            YIO_LOG_ERROR_EVENT("ConfigApplyingState.UnexpectedNetworkPreference", "Unexpected network_preference value: " << v);
        }
    }

    machine_->rtcConfig_ = rtcConfig;

    if (config_.isMember("mediasession_configuration")) {
        const auto& ms = config_["mediasession_configuration"];

        if (ms.isMember("keepalive_interval")) {
            machine_->keepaliveInterval_ = std::chrono::seconds(ms["keepalive_interval"].asInt());
        }
    }

    std::unique_ptr<messenger::rtc::FakeWebRtcVideoEncoderFactory> fakeVideoEncoderFactory(new messenger::rtc::FakeWebRtcVideoEncoderFactory());
    std::unique_ptr<messenger::rtc::FakeWebRtcVideoDecoderFactory> fakeVideoDecoderFactory(new messenger::rtc::FakeWebRtcVideoDecoderFactory());

    fakeVideoEncoderFactory->AddSupportedVideoCodecType("VP9");
    fakeVideoEncoderFactory->AddSupportedVideoCodecType("H264");
    fakeVideoEncoderFactory->AddSupportedVideoCodecType("VP8");

    fakeVideoDecoderFactory->AddSupportedVideoCodecType("VP9");
    fakeVideoDecoderFactory->AddSupportedVideoCodecType("H264");
    fakeVideoDecoderFactory->AddSupportedVideoCodecType("VP8");

    machine_->peerConnectionFactory_ = webrtc::CreatePeerConnectionFactory(
        nullptr,
        machine_->webrtcWorkerThread_.get(),
        machine_->webrtcSignalingThread_.get(),
        machine_->audio_,
        webrtc::CreateBuiltinAudioEncoderFactory(),
        webrtc::CreateBuiltinAudioDecoderFactory(),
        std::move(fakeVideoEncoderFactory),
        std::move(fakeVideoDecoderFactory),
        nullptr,
        machine_->apm_);

    if (!machine_->peerConnectionFactory_) {
        YIO_LOG_ERROR_EVENT("ConfigApplyingState.NullPeerConnectionFactory", "Can't create PeerConnectionFactory");

        machine_->setState(std::make_unique<PeerConnectionCreationFailedState>(machine_));

    } else {
        machine_->setState(std::make_unique<PeerConnectionCreatingState>(machine_));
    }
}

void SessionStateMachine::ConfigApplyingState::exit() {
}
