#include "local_offer_creating_state.h"

#include "local_offer_sending_state.h"
#include "local_offer_setting_state.h"

#include <contrib/libs/webrtc/api/peer_connection_interface.h>
#include <contrib/libs/webrtc/api/rtp_transceiver_interface.h>

#include <memory>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger::rtc;

void SessionStateMachine::LocalOfferCreatingState::Observer::OnSuccess(webrtc::SessionDescriptionInterface* desc) {
    std::string sdp;
    if (!desc->ToString(&sdp)) {
        // Should never happen since offer is created by webrtc
        throw std::runtime_error("Can't serialize offer");
    }

    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref, sdp]() {
        if (inState_) {
            if (machine_->direction_ == Direction::OUTGOING) {
                machine_->setState(std::make_unique<LocalOfferSettingState>(
                    machine_, sdp));

            } else {
                machine_->setState(std::make_unique<LocalOfferSendingState>(machine_, sdp));
            }

        } else {
            YIO_LOG_INFO("Already left " << LocalOfferCreatingState::NAME);
        }
    });
}

void SessionStateMachine::LocalOfferCreatingState::Observer::OnFailure(
    webrtc::RTCError error) {
    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref, error]() {
        if (inState_) {
            YIO_LOG_ERROR_EVENT("LocalOfferCreatingState.CreateOfferFailed", "Can't create offer: " << error.message());

            machine_->notifyFailure(error.message());

            machine_->setState(std::make_unique<LocalOfferCreatingState>(machine_, true));
        }
    });
}

const std::string SessionStateMachine::LocalOfferCreatingState::NAME = "LocalOfferCreatingState";

SessionStateMachine::LocalOfferCreatingState::LocalOfferCreatingState(SessionStateMachine* machine, bool performIceRestart)
    : NegotiatingState(machine)
    , performIceRestart_(performIceRestart)
    , observer_(new ::rtc::RefCountedObject<Observer>(machine))
{
}

void SessionStateMachine::LocalOfferCreatingState::enter() {
    NegotiatingState::enter();

    observer_->enter();

    machine_->candidatesSender_->reset();

    webrtc::PeerConnectionInterface::RTCOfferAnswerOptions opts;

    if (machine_->rtcConfig_.sdp_semantics == webrtc::SdpSemantics::kPlanB) {
        opts.offer_to_receive_audio = webrtc::PeerConnectionInterface::RTCOfferAnswerOptions::kOfferToReceiveMediaTrue;
        opts.offer_to_receive_video = webrtc::PeerConnectionInterface::RTCOfferAnswerOptions::kOfferToReceiveMediaTrue;
    }

    if (performIceRestart_) {
        opts.ice_restart = true;
    }

    // TODO: addPlaceholderVideoTransceiver

    machine_->peerConnection_->CreateOffer(observer_, opts);
}

void SessionStateMachine::LocalOfferCreatingState::exit() {
    observer_->exit();

    NegotiatingState::exit();
}

bool SessionStateMachine::LocalOfferCreatingState::hasPlaceholderVideoTransceiver() const {
    for (const auto& t : machine_->peerConnection_->GetTransceivers()) {
        if (t->media_type() != cricket::MEDIA_TYPE_VIDEO) {
            continue;
        }

        if (t->direction() == webrtc::RtpTransceiverDirection::kRecvOnly) {
            return true;
        }
    }

    return false;
}

void SessionStateMachine::LocalOfferCreatingState::addPlaceholderVideoTransceiver() {
    webrtc::RtpTransceiverInit t;
    t.direction = webrtc::RtpTransceiverDirection::kRecvOnly;

    machine_->peerConnection_->AddTransceiver(cricket::MEDIA_TYPE_VIDEO, t);
}
