#include "local_offer_setting_state.h"

#include "local_offer_sending_state.h"
#include "remote_answer_setting_state.h"

#include <memory>

using namespace messenger::rtc;

void SessionStateMachine::LocalOfferSettingState::Observer::OnSuccess() {
    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref]() {
        if (inState_) {
            if (machine_->direction_ == Direction::OUTGOING) {
                machine_->setState(std::make_unique<LocalOfferSendingState>(
                    machine_, sdp_));

            } else {
                machine_->setState(std::make_unique<RemoteAnswerSettingState>(machine_, answer_));
            }

        } else {
            YIO_LOG_INFO("Already left " << LocalOfferSettingState::NAME);
        }
    });
}

void SessionStateMachine::LocalOfferSettingState::Observer::OnFailure(
    webrtc::RTCError error) {
    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref, error]() {
        if (inState_) {
            machine_->notifyFailure(error.message());

            machine_->mediator_->reportError(
                machine_->sessionUuid_,
                error.message(),
                MediatorApi::ErrorType::INVALID_OFFER);

        } else {
            YIO_LOG_INFO("Already left " << LocalOfferSettingState::NAME);
        }
    });
}

const std::string SessionStateMachine::LocalOfferSettingState::NAME = "LocalOfferSettingState";

SessionStateMachine::LocalOfferSettingState::LocalOfferSettingState(
    SessionStateMachine* machine,
    const std::string& sdp,
    const std::string& remoteAnswer)
    : NegotiatingState(machine)
    , sdp_(sdp)
    , answer_(remoteAnswer)
    , observer_(new ::rtc::RefCountedObject<Observer>(machine_, sdp_, answer_))
{
}

void SessionStateMachine::LocalOfferSettingState::enter() {
    NegotiatingState::enter();

    observer_->enter();

    webrtc::SdpParseError err;
    std::unique_ptr<webrtc::SessionDescriptionInterface> description(webrtc::CreateSessionDescription("offer", sdp_, &err));

    if (!description) {
        // Should never happen since offer is created by webrtc
        throw std::runtime_error("Can't parse offer");
    }

    machine_->peerConnection_->SetLocalDescription(observer_, description.release());
}

void SessionStateMachine::LocalOfferSettingState::exit() {
    observer_->exit();

    NegotiatingState::exit();
}
