#include "negotiating_state.h"

#include "local_offer_creating_state.h"
#include "peer_connection_broken_state.h"
#include "remote_offer_setting_state.h"
#include "renegotiation_needed_state.h"

#include <yandex_io/callkit/rtc/media/entities.h>

#include <memory>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger::rtc;

SessionStateMachine::NegotiatingState::NegotiatingState(SessionStateMachine* machine)
    : SessionState(machine)
{
}

void SessionStateMachine::NegotiatingState::enter() {
    machine_->subscribe(this);

    mediatorSubscription_ = machine_->mediator_->subscribe([this](const MediatorApi::Message& message) {
        if (!message.isMember("method")) {
            return;
        }

        const std::string& method = message["method"].asString();

        if (method == "addCandidates") {
            handleRemoteCandidatesAddition(message);

        } else if (method == "removeCandidates") {
            handleRemoteCandidatesRemoving(message);

        } else if (method == "offer") {
            handleRemoteOffer(message);

        } else if (method == "updateState") {
            handleRemoteMediaStateUpdate(message);
        }
    });
}

void SessionStateMachine::NegotiatingState::exit() {
    mediatorSubscription_.reset(nullptr);

    machine_->unsubscribe(this);
}

void SessionStateMachine::NegotiatingState::acknowledge(const std::string& transactionId) {
    machine_->mediator_->acknowledge(machine_->sessionUuid_, transactionId);
}

void SessionStateMachine::NegotiatingState::handleRemoteCandidatesAddition(const MediatorApi::Message& message) {
    if (!message.isMember("params")) {
        return;
    }

    const auto& params = message["params"];

    if (!params.isMember("guid")) {
        return;
    }

    if (!params.isMember("candidates") || params["candidates"].empty()) {
        return;
    }

    const std::string& guid = params["guid"].asString();

    if (guid != machine_->sessionUuid_) {
        return;
    }

    std::vector<IceCandidate> failed;

    for (const auto& it : params["candidates"]) {
        IceCandidate c;

        if (it.isMember("m_id")) {
            c.mId = it["m_id"].asString();
        }

        if (it.isMember("m_line_index")) {
            c.mLineIndex = it["m_line_index"].asInt();
        }

        if (it.isMember("candidate")) {
            c.candidate = it["candidate"].asString();
        }

        if (c.candidate.empty()) {
            // TODO
            continue;
        }

        YIO_LOG_INFO("Add candidate " << c.candidate);

        webrtc::SdpParseError error;
        std::unique_ptr<webrtc::IceCandidateInterface> candidate(
            webrtc::CreateIceCandidate(c.mId, c.mLineIndex, c.candidate, &error));

        if (!candidate || !machine_->peerConnection_->AddIceCandidate(candidate.get())) {
            failed.push_back(c);
        }
    }

    if (failed.empty()) {
        acknowledge(message["id"].asString());

    } else {
        std::string details("Can't add candidates: ");
        for (const IceCandidate& c : failed) {
            details += c.candidate + ";"; // TODO: is it same as java format?
        }
        YIO_LOG_ERROR_EVENT("NegotiatingState.AddCandidatesFailed", details);

        machine_->mediator_->returnError(
            machine_->sessionUuid_,
            message["id"].asString(),
            "Can't add candidate",
            details,
            MediatorApi::ErrorType::ADD_CANDIDATE_FAILED);
    }
}

void SessionStateMachine::NegotiatingState::handleRemoteCandidatesRemoving(const MediatorApi::Message& message) {
    if (!message.isMember("params")) {
        return;
    }

    const auto& params = message["params"];

    if (!params.isMember("guid")) {
        return;
    }

    if (!params.isMember("candidates") || params["candidates"].empty()) {
        return;
    }

    const std::string& guid = params["guid"].asString();

    if (guid != machine_->sessionUuid_) {
        return;
    }

    acknowledge(message["id"].asString());

    std::vector<cricket::Candidate> candidates;

    for (const auto& jc : params["candidates"]) {
        IceCandidate c;

        if (jc.isMember("m_id")) {
            c.mId = jc["m_id"].asString();
        }

        if (jc.isMember("m_line_index")) {
            c.mLineIndex = jc["m_line_index"].asInt();
        }

        if (jc.isMember("candidate")) {
            c.candidate = jc["candidate"].asString();
        }

        webrtc::SdpParseError error;
        std::unique_ptr<webrtc::IceCandidateInterface> candidate(
            webrtc::CreateIceCandidate(c.mId, c.mLineIndex, c.candidate, &error));

        candidates.push_back(candidate->candidate());
    }

    if (!machine_->peerConnection_->RemoveIceCandidates(candidates)) {
        machine_->mediator_->reportError(
            machine_->sessionUuid_,
            "Can't remove candidates",
            MediatorApi::ErrorType::REMOVE_CANDIDATES_FAILED);
    }
}

void SessionStateMachine::NegotiatingState::handleRemoteOffer(const MediatorApi::Message& message) {
    if (!message.isMember("params")) {
        return;
    }

    const auto& params = message["params"];

    if (!params.isMember("guid")) {
        return;
    }

    const std::string& guid = params["guid"].asString();

    if (guid != machine_->sessionUuid_) {
        return;
    }

    acknowledge(message["id"].asString());

    // handleRemoteOffer

    if (!params.isMember("offer")) {
        return;
    }

    const bool shouldSetRemoteOffer = machine_->direction_ == Direction::INCOMING || !negotiatingLocalOffer();

    if (shouldSetRemoteOffer) {
        machine_->setState(std::make_unique<RemoteOfferSettingState>(machine_, params["offer"].asString()));

    } else {
        // XXX: performIceRestart is false in android state machine
        machine_->setState(std::make_unique<LocalOfferCreatingState>(machine_, true));
    }
}

void SessionStateMachine::NegotiatingState::handleRemoteMediaStateUpdate(const MediatorApi::Message& message) {
    // TODO
}

void SessionStateMachine::NegotiatingState::OnIceConnectionChange(webrtc::PeerConnectionInterface::IceConnectionState newState) {
    std::string st;

    switch (newState) {
        case webrtc::PeerConnectionInterface::kIceConnectionNew:
            st = "new";
            break;
        case webrtc::PeerConnectionInterface::kIceConnectionChecking:
            st = "checking";
            break;
        case webrtc::PeerConnectionInterface::kIceConnectionConnected:
            st = "connected";
            break;
        case webrtc::PeerConnectionInterface::kIceConnectionCompleted:
            st = "completed";
            break;
        case webrtc::PeerConnectionInterface::kIceConnectionFailed:
            st = "failed";
            break;
        case webrtc::PeerConnectionInterface::kIceConnectionDisconnected:
            st = "disconnected";
            break;
        case webrtc::PeerConnectionInterface::kIceConnectionClosed:
            st = "closed";
            break;
        default:
            // Should never happen
            throw std::runtime_error("Unexpected state");
    }

    Json::Value event;
    event["ice_connection_state"] = st;
    event["type"] = "iceConnectionStateChange";

    machine_->mediator_->reportEvent(machine_->sessionUuid_, event);

    if (machine_->peerConnection_->ice_gathering_state() == webrtc::PeerConnectionInterface::kIceGatheringComplete) {
        if (newState == webrtc::PeerConnectionInterface::kIceConnectionFailed) {
            machine_->setState(std::make_unique<PeerConnectionBrokenState>(machine_));
        }
    }

    if (newState == webrtc::PeerConnectionInterface::kIceConnectionCompleted) {
        machine_->statsSender_->sendImmediately();
    }
}

void SessionStateMachine::NegotiatingState::OnRenegotiationNeeded() {
    if (!shouldSkipRenegotiation()) {
        machine_->setState(std::make_unique<RenegotiationNeededState>(machine_));

    } else {
        YIO_LOG_INFO("Skip renegotiating");
    }
}

void SessionStateMachine::NegotiatingState::OnIceCandidate(const webrtc::IceCandidateInterface* candidate) {
    IceCandidate c;
    c.mId = candidate->sdp_mid();
    c.mLineIndex = candidate->sdp_mline_index();

    if (!candidate->ToString(&c.candidate)) {
        // Should never happen since candidate is provided by webrtc
        throw std::runtime_error("Can't serialize ice candidate");
    }

    machine_->candidatesSender_->addCandidate(c);
}

void SessionStateMachine::NegotiatingState::OnIceCandidatesRemoved(const std::vector<cricket::Candidate>& candidates) {
    YIO_LOG_WARN("OnIceCandidatesRemoved");
}

void SessionStateMachine::NegotiatingState::OnIceGatheringChange(webrtc::PeerConnectionInterface::IceGatheringState newState) {
    if (newState == webrtc::PeerConnectionInterface::kIceGatheringComplete) {
        machine_->candidatesSender_->completeTrickling();
    }
}

void SessionStateMachine::NegotiatingState::OnAddTrack(
    ::rtc::scoped_refptr<webrtc::RtpReceiverInterface> receiver,
    const std::vector<::rtc::scoped_refptr<webrtc::MediaStreamInterface>>& /*streams*/)
{
    YIO_LOG_WARN("OnAddTrack");
}

void SessionStateMachine::NegotiatingState::OnSignalingChange(webrtc::PeerConnectionInterface::SignalingState /*new_state*/) {
}

void SessionStateMachine::NegotiatingState::OnDataChannel(::rtc::scoped_refptr<webrtc::DataChannelInterface> /*data_channel*/) {
}
