#include "remote_answer_receiving_state.h"

#include "local_offer_sending_state.h"
#include "local_offer_setting_state.h"
#include "remote_answer_setting_state.h"

#include <memory>

using namespace messenger::rtc;

const std::string SessionStateMachine::RemoteAnswerReceivingState::NAME = "RemoteAnswerReceivingState";

SessionStateMachine::RemoteAnswerReceivingState::RemoteAnswerReceivingState(SessionStateMachine* machine, const std::string& sdp)
    : NegotiatingState(machine)
    , sdp_(sdp)
{
}

void SessionStateMachine::RemoteAnswerReceivingState::enter() {
    NegotiatingState::enter();

    subscription_ = machine_->mediator_->subscribe([this](const MediatorApi::Message& message) {
        if (message["method"] == "answer") {
            handleRemoteAnswer(message);
        }
    });

    // TODO: how does android state machine work without this retry code?
    timer_ = machine_->timerFactory_->createTimer(std::chrono::seconds(15), [this]() {
        machine_->setState(std::make_unique<LocalOfferSendingState>(machine_, sdp_));
    });
}

void SessionStateMachine::RemoteAnswerReceivingState::exit() {
    timer_->cancel();

    subscription_.reset(nullptr);

    NegotiatingState::exit();
}

void SessionStateMachine::RemoteAnswerReceivingState::handleRemoteAnswer(const MediatorApi::Message& message) {
    if (!message.isMember("params")) {
        return;
    }

    const auto& params = message["params"];

    if (!params.isMember("guid")) {
        return;
    }

    const std::string& guid = params["guid"].asString();

    if (guid != machine_->sessionUuid_) {
        return;
    }

    machine_->mediator_->acknowledge(guid, message["id"].asString());

    if (!params.isMember("answer")) {
        return;
    }

    if (machine_->direction_ == Direction::OUTGOING) {
        machine_->setState(std::make_unique<RemoteAnswerSettingState>(machine_, params["answer"].asString()));

    } else {
        machine_->setState(std::make_unique<LocalOfferSettingState>(machine_, sdp_, params["answer"].asString()));
    }
}
