#include "remote_answer_setting_state.h"

#include "waiting_for_connection_state.h"

#include <memory>

using namespace messenger::rtc;

void SessionStateMachine::RemoteAnswerSettingState::Observer::OnSuccess() {
    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref]() {
        if (inState_) {
            machine_->setState(std::make_unique<WaitingForConnectionState>(
                machine_));

        } else {
            YIO_LOG_INFO("Already left " << RemoteAnswerSettingState::NAME);
        }
    });
}

void SessionStateMachine::RemoteAnswerSettingState::Observer::OnFailure(
    webrtc::RTCError error) {
    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref, error]() {
        if (inState_) {
            machine_->notifyFailure(error.message());

            machine_->mediator_->reportError(
                machine_->sessionUuid_,
                error.message(),
                MediatorApi::ErrorType::INVALID_ANSWER);

        } else {
            YIO_LOG_INFO("Already left " << RemoteAnswerSettingState::NAME);
        }
    });
}

const std::string SessionStateMachine::RemoteAnswerSettingState::NAME = "RemoteAnswerSettingState";

SessionStateMachine::RemoteAnswerSettingState::RemoteAnswerSettingState(SessionStateMachine* machine, const std::string& sdp)
    : NegotiatingState(machine)
    , sdp_(sdp)
    , observer_(new ::rtc::RefCountedObject<Observer>(machine_))
{
}

void SessionStateMachine::RemoteAnswerSettingState::enter() {
    NegotiatingState::enter();

    observer_->enter();

    webrtc::SdpParseError error;
    std::unique_ptr<webrtc::SessionDescriptionInterface> description(webrtc::CreateSessionDescription("answer", sdp_, &error));

    if (!description) {
        machine_->notifyFailure(error.line + ": " + error.description);

        machine_->mediator_->reportError(
            machine_->sessionUuid_,
            error.line + ": " + error.description,
            MediatorApi::ErrorType::INVALID_ANSWER);
    }

    machine_->peerConnection_->SetRemoteDescription(observer_, description.release());
}

void SessionStateMachine::RemoteAnswerSettingState::exit() {
    observer_->exit();

    NegotiatingState::exit();
}
