#include "remote_offer_setting_state.h"

#include "local_answer_creating_state.h"

#include <memory>

using namespace messenger::rtc;

void SessionStateMachine::RemoteOfferSettingState::Observer::OnSuccess() {
    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref]() {
        if (inState_) {
            machine_->setState(std::make_unique<LocalAnswerCreatingState>(
                machine_));

        } else {
            YIO_LOG_INFO("Already left " << RemoteOfferSettingState::NAME);
        }
    });
}

void SessionStateMachine::RemoteOfferSettingState::Observer::OnFailure(
    webrtc::RTCError error) {
    ::rtc::scoped_refptr<Observer> ref(this);

    machine_->workerThread_->execute([this, ref, error]() {
        if (inState_) {
            machine_->notifyFailure(error.message());

            machine_->mediator_->reportError(
                machine_->sessionUuid_, error.message(),
                MediatorApi::ErrorType::INVALID_OFFER);

        } else {
            YIO_LOG_INFO("Already left " << RemoteOfferSettingState::NAME);
        }
    });
}

const std::string SessionStateMachine::RemoteOfferSettingState::NAME =
    "RemoteOfferSettingState";

SessionStateMachine::RemoteOfferSettingState::RemoteOfferSettingState(
    SessionStateMachine* machine, const std::string& offer)
    : NegotiatingState(machine)
    , offer_(offer)
    , observer_(new ::rtc::RefCountedObject<Observer>(machine_))
{
}

void SessionStateMachine::RemoteOfferSettingState::enter() {
    NegotiatingState::enter();

    observer_->enter();

    machine_->candidatesSender_->reset();

    webrtc::SdpParseError error;
    std::unique_ptr<webrtc::SessionDescriptionInterface> description(
        webrtc::CreateSessionDescription("offer", offer_, &error));

    if (!description) {
        machine_->notifyFailure(error.line + ": " + error.description);

        machine_->mediator_->reportError(machine_->sessionUuid_,
                                         error.line + ": " + error.description,
                                         MediatorApi::ErrorType::INVALID_OFFER);
    }

    machine_->peerConnection_->SetRemoteDescription(observer_,
                                                    description.release());
}

void SessionStateMachine::RemoteOfferSettingState::exit() {
    observer_->exit();

    NegotiatingState::exit();
}
