#pragma once

#include <yandex_io/callkit/calls/call/notifier.h>
#include <yandex_io/callkit/calls/call/status.h>
#include <yandex_io/callkit/rtc/media/direction.h>
#include <yandex_io/callkit/util/observer_list.h>

#include <string>

namespace messenger {

    class Session {
    public:
        virtual ~Session() = default;

    public:
        // Call control

        virtual void startCall(const std::string& userGuid, const std::string& callPayload) = 0;

        virtual void startCallToOwnDevice(const std::string& deviceId, const std::string& callPayload) = 0;

        virtual void declineIncomingCall() = 0;

        virtual void acceptIncomingCall() = 0;

        virtual void hangupCall() = 0;

        virtual void sendHeartbeat() = 0;

        // When True, IncomingCall without ReceiverDeviceId ('usual calls', not
        // directed to a device) are accepted.
        // When False, those calls are completely ignored.
        //
        // Note that ALWAYS ignored:
        // - IncomingCall with 'my' ReceiverDeviceId but from 'foreign' users
        // - IncomingCall with 'foreign' ReceiverDeviceId
        //
        //                         ReceiverDeviceId
        // +-----------------+-----------------------+-------------------+-----------------------+
        // |                 |      no device id     |   this device id  |   another device id   |
        // +-----------------+-----------------------+-------------------+-----------------------+
        // | from this user  | NO                    | YES               | NO                    |
        // |                 | refused by backend    |                   | routed by backend     |
        // |                 |                       |                   | and ignored by client |
        // +-----------------+-----------------------+-------------------+-----------------------+
        // | from other user | 'seAllowUsualCalls'   | NO                | NO                    |
        // |                 | client-side decision  | ignored by client | routed by backend     |
        // |                 |                       |                   | and ignored by client |
        // +-----------------+-----------------------+-------------------+-----------------------+
        virtual void setAllowUsualCalls(bool allowUsualCalls) = 0;

    public:
        // State

        struct State {
            // Session
            const bool authorized;
            const bool connected;

            // Call
            const Status status;
            const rtc::Direction direction;
            const std::string callGuid;
            const std::string userName;
            const std::string userAvatar;
            const std::string userGuid;
            const std::string callerDeviceId;
            const std::string callerPayload;
            const bool isSelfCall;
            const bool tokenExpired;

            bool operator==(const State& other) const;
        };

        virtual State getState() = 0;

    public:
        // Call state changed subscription
        using StateChangedCallback = std::function<void(const State&)>;
        using StateChangedSubscription = ObserverList<const State&>::ScopedSubscription;
        StateChangedSubscription subscribeStateChanged(StateChangedCallback callback);

        // Call accepted subscription
        using CallAcceptedCallback = messenger::Notifier::OnAcceptListener;
        using CallAcceptedSubscription = messenger::Notifier::OnAcceptSubscription;
        CallAcceptedSubscription subscribeCallAccepted(CallAcceptedCallback callback);

        // Call declined subscription
        using CallDeclinedCallback = messenger::Notifier::OnDeclineListener;
        using CallDeclinedSubscription = messenger::Notifier::OnDeclineSubscription;
        CallDeclinedSubscription subscribeCallDeclined(CallDeclinedCallback callback);

        // Call failed subscription
        using CallFailedCallback = messenger::Notifier::OnFailureListener;
        using CallFailedSubscription = messenger::Notifier::OnFailureSubscription;
        CallFailedSubscription subscribeCallFailed(CallFailedCallback callback);

        // Call started subscription
        using CallStartedCallback = messenger::Notifier::OnStartListener;
        using CallStartedSubscription = messenger::Notifier::OnStartSubscription;
        CallStartedSubscription subscribeCallStarted(CallStartedCallback callback);

        // Call ended subscription
        using CallEndedCallback = messenger::Notifier::OnEndListener;
        using CallEndedSubscription = messenger::Notifier::OnEndSubscription;
        CallEndedSubscription subscribeCallEnded(CallEndedCallback callback);

        // Call creation failed subscription
        CallbackSubscription subscribeCallCreationFailed(Callback callback);

    protected:
        ObserverList<const State&> stateObserverList_;
        ObserverList<const std::string&> acceptObserverList_;
        ObserverList<const std::string&> declineObserverList_;
        ObserverList<const std::string&, CallTransport::ErrorCode, const std::string&> failedObserverList_;
        ObserverList<const std::string&> startObserverList_;
        ObserverList<const std::string&> endObserverList_;
        CallbackObserverList callCreationFailedObserverList_;
    };

} // namespace messenger
