#include "session_impl.h"

#include "session_service.h"
#include "yandex_io/callkit/util/weak_utils.h"

#include <functional>
#include <future>
#include <memory>

using namespace messenger;

void SessionImpl::init(const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory, std::shared_ptr<YandexIO::IDevice> device, const SessionParams& params) {
    Y_VERIFY(!params.token.empty());
    Y_VERIFY(!params.passportUid.empty());

    workerThread_ = LoopThread::create();

    auto onTokenExpiredCallback = [weakThis = weak_from(this)] {
        if (auto s = weakThis.lock()) {
            s->onTokenExpired();
        }
    };

    workerThread_->execute([=]() {
        pthread_setname_np(pthread_self(), "MessengerServiceWorker");
        sessionService_ = SessionService::create(
            ipcFactory, onTokenExpiredCallback, params, workerThread_, device);
        sessionService_->init();

        auto call = sessionService_->getCallHolder()->getCall();

        onCurrentCallChanged(call.get());

        callChangedSubscription_ = sessionService_->getCallHolder()->onCallChanged.subscribe(
            [this](std::shared_ptr<Call> call) {
                onCurrentCallChanged(call.get());
            });

        callCreationFailedSubscription_ = sessionService_->getCallHolder()->onCallCreationFailed.subscribe(
            [this](const std::string& /*userguid*/, CallTransport::ErrorCode /*errCode*/) {
                callCreationFailedObserverList_.notifyObservers();
            });
    });
}

SessionImpl::~SessionImpl() {
    workerThread_->execute([this] { sessionService_.reset(); });
    workerThread_->destroyBlocked();
    Y_VERIFY(!sessionService_);
}

void SessionImpl::waitForIdle() {
    std::function<void()> tryUnlock = [this, &tryUnlock]() {
        if (sessionService_->hasBgTasks()) {
            sessionService_->runOnBgIdle(tryUnlock);
        } else if (workerThread_->isIdle()) {
            std::unique_lock<std::mutex> lck(idleMutex_);
            idleCheck_ = true;
            lck.unlock();
            idleCondition_.notify_one();
        } else {
            workerThread_->executeOnIdle(tryUnlock);
        }
    };
    std::unique_lock<std::mutex> lock(idleMutex_);
    idleCheck_ = false;
    workerThread_->executeOnIdle(tryUnlock);
    idleCondition_.wait(lock, [this] { return idleCheck_.load(); });
}

Session::State SessionImpl::getState() {
    std::promise<Session::State> promise;
    auto future = promise.get_future();

    workerThread_->execute([this, &promise]() {
        promise.set_value(getStateImpl());
    });

    return future.get();
}

Session::State SessionImpl::getStateImpl() const {
    auto call = sessionService_->getCallHolder()->getCall();

    if (call) {
        return {
            sessionService_->isAuthorized(),
            sessionService_->isConnected(),
            call->getStatus(),
            call->getDirection(),
            call->getCallGuid(),
            call->getUserName(),
            call->getUserAvatar(),
            call->getUserGuid(),
            call->getCallerDeviceId(),
            call->getCallerPayload(),
            sessionService_->getOwnGuid() == call->getUserGuid(),
            sessionService_->isTokenExpired(),
        };

    } else {
        return {
            sessionService_->isAuthorized(),
            sessionService_->isConnected(),
            Status::NOCALL,
            rtc::Direction::INCOMING,
            "",
            "",
            "",
            "",
            "",
            "",
            false,
            sessionService_->isTokenExpired(),
        };
    }
}

void SessionImpl::onCurrentCallChanged(Call* call) {
    callStateSubscription_.reset();
    callAcceptSubscription_.reset();
    callDeclineSubscription_.reset();
    callFailureSubscription_.reset();
    callStartSubscription_.reset();
    callEndSubscription_.reset();

    if (call) {
        onCallStatusChanged(call);

        callStateSubscription_ = call->notifier->onStatusChangeListeners.subscribe(
            call,
            [this, call](const std::string& /*unused*/) {
                onCallStatusChanged(call);
            });

        callAcceptSubscription_ = call->notifier->subscribeOnAccept(
            [this](const std::string& chatId) {
                acceptObserverList_.notifyObservers(chatId);
            });

        callDeclineSubscription_ = call->notifier->subscribeOnDecline(
            [this](const std::string& chatId) {
                declineObserverList_.notifyObservers(chatId);
            });

        callFailureSubscription_ = call->notifier->subscribeOnFailure(
            [this](const std::string& chatId, messenger::CallTransport::ErrorCode code, const std::string& msg) {
                failedObserverList_.notifyObservers(chatId, code, msg);
            });

        callStartSubscription_ = call->notifier->subscribeOnStart(
            [this](const std::string& chatId) {
                startObserverList_.notifyObservers(chatId);
            });

        callEndSubscription_ = call->notifier->subscribeOnEnd(
            [this](const std::string& chatId) {
                endObserverList_.notifyObservers(chatId);
            });

    } else {
        stateObserverList_.notifyObservers(Session::State{
            sessionService_->isAuthorized(),
            sessionService_->isConnected(),
            Status::NOCALL,
            rtc::Direction::INCOMING,
            "",
            "",
            "",
            "",
            "",
            "",
            false,
            sessionService_->isTokenExpired(),
        });
    }
}

void SessionImpl::onCallStatusChanged(Call* call) {
    stateObserverList_.notifyObservers(Session::State{
        sessionService_->isAuthorized(),
        sessionService_->isConnected(),
        call->getStatus(),
        call->getDirection(),
        call->getCallGuid(),
        call->getUserName(),
        call->getUserAvatar(),
        call->getUserGuid(),
        call->getCallerDeviceId(),
        call->getCallerPayload(),
        sessionService_->getOwnGuid() == call->getUserGuid(),
        sessionService_->isTokenExpired(),
    });
}

void SessionImpl::onTokenExpired() {
    stateObserverList_.notifyObservers(Session::State{
        sessionService_->isAuthorized(),
        sessionService_->isConnected(),
        Status::NOCALL,
        rtc::Direction::INCOMING,
        "",
        "",
        "",
        "",
        "",
        "",
        false,
        sessionService_->isTokenExpired(),
    });
}

void SessionImpl::setAllowUsualCalls(bool allowUsualCalls) {
    workerThread_->execute(
        [=]() { sessionService_->setAllowUsualCalls(allowUsualCalls); });
}

void SessionImpl::sendHeartbeat() {
    workerThread_->execute([=]() { sessionService_->sendHeartbeat(); });
}

void SessionImpl::startCall(const std::string& userGuid, const std::string& callPayload) {
    workerThread_->execute([=]() { sessionService_->startCall(userGuid, callPayload); });
}

void SessionImpl::startCallToOwnDevice(const std::string& deviceId, const std::string& callPayload) {
    workerThread_->execute([=]() { sessionService_->startCallToOwnDevice(deviceId, callPayload); });
}

void SessionImpl::declineIncomingCall() {
    workerThread_->execute([=] { sessionService_->declineIncomingCall(); });
}

void SessionImpl::acceptIncomingCall() {
    workerThread_->execute([=] { sessionService_->acceptIncomingCall(); });
}

void SessionImpl::hangupCall() {
    workerThread_->execute([=] { sessionService_->hangupCall(); });
}
