#include "session_provider.h"

#include "session.h"
#include "session_impl.h"
#include "session_settings.h"

#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_connector.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <memory>
#include <tuple>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

SessionProvider::SessionProvider(
    std::shared_ptr<quasar::ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<YandexIO::IDevice> device,
    const std::string& appId,
    const std::string& appName,
    const std::string& appVersion,
    const std::string& passportUid,
    const std::string& authToken,
    const std::vector<std::string>& xivaSubscriptions)
    : ipcFactory_(std::move(ipcFactory))
    , device_(std::move(device))
    , passportUid_(passportUid)
    , authToken_(authToken)
    , xivaSubscriptions_(xivaSubscriptions)
{
    for (const std::string& service : xivaSubscriptions_) {
        if (service == "messenger") {
            configEnv_ = "alpha";
            break;
        }
        if (service == "messenger-prod") {
            configEnv_ = "production";
            break;
        }
    }

    struct PlatformInfoProviderImpl: public PlatformInfoProvider {
        PlatformInfo base;
        const std::shared_ptr<YandexIO::IDevice> device;
        PlatformInfoProviderImpl(PlatformInfo base, std::shared_ptr<YandexIO::IDevice> dev)
            : base(base)
            , device(std::move(dev))
        {
        }
        PlatformInfo create() override {
            PlatformInfo info(base);
            info.deviceId = device->deviceId();
            const auto& common = device->configuration()->getServiceConfig("common");
            info.osVersion =
                quasar::tryGetString(common, "os_version", "unknown");
            info.deviceManufacturer =
                quasar::tryGetString(common, "device_manufacturer", "unknown");
            info.deviceType =
                quasar::tryGetString(common, "deviceType", "unknown");
            info.deviceOs = quasar::tryGetString(common, "os", "unknown");
            info.softwareVersion =
                quasar::tryGetString(common, "softwareVersion", "unknown");
            return info;
        }
    };

    PlatformInfo base;
    base.appId = appId;
    base.appName = appName;
    base.appVersion = appVersion;

    info_ = std::make_shared<PlatformInfoProviderImpl>(base, device_);
}

std::shared_ptr<Session> SessionProvider::createCustomSession(
    const std::string& passportUid,
    const std::string& token,
    const std::string& server,
    bool useDedicatedXiva,
    const std::string& xivaServiceName,
    const std::string& messengerApiHost,
    const std::vector<std::string>& xivaSubscriptions)
{
    SessionParams sessionParams(
        passportUid,
        token,
        server,
        useDedicatedXiva,
        xivaServiceName,
        messengerApiHost,
        xivaSubscriptions);

    if (sessionParams.token.empty() || sessionParams.passportUid.empty()) {
        throw std::runtime_error("Invalid params");
    }

    sessionParams.platformInfo = info_;

    return SessionImpl::create(ipcFactory_, device_, sessionParams);
}

std::shared_ptr<Session> SessionProvider::createQuasarPushdSession(
    const std::string& passportUid,
    const std::string& authToken)
{
    if (passportUid.empty() || authToken.empty()) {
        return std::unique_ptr<Session>();
    }

    if (configEnv_.empty()) {
        throw std::runtime_error(
            "Invalid configuration for Pushd.\nAdd in "
            "'/system/vendor/quasar/quasar.cfg'  'pushd/xivaServices':\n"
            " - 'messenger' for alpha or\n"
            " - 'messenger-prod' for prod");
    }

    SessionParams sessionParams;
    sessionParams.useDedicatedXiva = false;
    sessionParams.server = configEnv_ == "production" ? SessionParams::PRODUCTION : SessionParams::ALPHA;
    sessionParams.passportUid = passportUid;
    sessionParams.token = authToken;
    sessionParams.platformInfo = info_;
    sessionParams.xivaSubscriptions = xivaSubscriptions_;

    return SessionImpl::create(ipcFactory_, device_, sessionParams);
}

std::shared_ptr<Session> SessionProvider::createQuasarPushdSession() {
    std::lock_guard<std::mutex> lock(mutex_);

    return createQuasarPushdSession(passportUid_, authToken_);
}

std::shared_ptr<Session> SessionProvider::createQuasarDedicatedSession(const std::string& env) {
    std::lock_guard<std::mutex> lock(mutex_);

    if (passportUid_.empty() || authToken_.empty()) {
        return std::unique_ptr<Session>();
    }

    std::string envStr = env;

    if (envStr == "config") {
        if (configEnv_.empty()) {
            throw std::runtime_error(
                "Invalid configuration for Pushd.\nAdd in "
                "'/system/vendor/quasar/quasar.cfg' - 'pushd/xivaServices':\n"
                " - 'messenger' for alpha or\n"
                " - 'messenger-prod' for prod");
        }
        envStr = configEnv_;

    } else if (envStr.empty()) {
        envStr = configEnv_.empty() ? "production" : configEnv_;
    }

    SessionParams sessionParams;
    sessionParams.useDedicatedXiva = true;
    sessionParams.server = envStr == "production" ? SessionParams::PRODUCTION : SessionParams::ALPHA;
    sessionParams.passportUid = passportUid_;
    sessionParams.token = authToken_;
    sessionParams.platformInfo = info_;

    return SessionImpl::create(ipcFactory_, device_, sessionParams);
}

// static
std::vector<std::string> SessionProvider::getXivaServices(const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory) {
    auto toPushd = ipcFactory->createIpcConnector("pushd");

    toPushd->connectToService();
    if (!toPushd->waitUntilConnected(std::chrono::seconds(10))) {
        YIO_LOG_ERROR_EVENT("SessionProvider.ListXivaSubscribtions.ConnectToPushdTimeout", "Can't get service names: can't connect to pushd");
        return {};
    }

    quasar::proto::QuasarMessage request;
    request.mutable_list_xiva_subscriptions();

    quasar::ipc::SharedMessage response;
    try {
        response = toPushd->sendRequestSync(std::move(request), std::chrono::seconds(5));

    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("SessionProvider.ListXivaSubscribtions.RequestToPushdFailed", "Can't get service names: " << e.what());
        return {};
    }

    if (!response->has_xiva_subscriptions()) {
        YIO_LOG_ERROR_EVENT("SessionProvider.ListXivaSubscribtions.InvalidPushdResponse", "Can't get service names: no xiva_subscriptions in response");
        return {};
    }

    return {
        response->xiva_subscriptions().xiva_subscriptions().begin(),
        response->xiva_subscriptions().xiva_subscriptions().end(),
    };
}
