#include "session_service.h"

#include "session_settings.h"

#include <yandex_io/callkit/api/session_api_service.h>
#include <yandex_io/callkit/calls/calling_service.h>
#include <yandex_io/callkit/connection/connection_service.h>
#include <yandex_io/callkit/connection/request_handler.h>
#include <yandex_io/callkit/storage/data_storage.h>
#include <yandex_io/callkit/util/async_service.h>
#include <yandex_io/callkit/util/loop_thread.h>

#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/logging/logging.h>

#include <memory>

using namespace messenger;

void SessionService::init(const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory,
                          SessionRequestFactory::OnExpiredTokenCallback callback,
                          const SessionParams& params,
                          std::shared_ptr<LoopThread> workerThread,
                          std::shared_ptr<YandexIO::IDevice> device) {
    settings_ = std::make_unique<SessionSettings>(params);
    workerThread_ = std::move(workerThread);

    auto onTokenExpiredCallback = [weakThis = weak_from(this), callback = std::move(callback)] {
        if (auto s = weakThis.lock()) {
            s->tokenExpired_ = true;
            callback();
        }
    };

    dataStorage_ =
        std::make_shared<DataStorage>(settings_->storageId, workerThread_);
    apiService_ = std::make_shared<SessionApiService>(workerThread_,
                                                      std::move(onTokenExpiredCallback),
                                                      dataStorage_, *settings_, device);
    connectionService_ = std::make_unique<ConnectionService>(
        ipcFactory, *settings_, workerThread_, dataStorage_, device->telemetry());
    callingService_ = CallingService::create(
        ipcFactory, *settings_, workerThread_, dataStorage_, apiService_->getRequestFactory(),
        apiService_->getTaskExecutor(), connectionService_->getRequestHandler(),
        connectionService_->getPushHandler(), device->configuration()->getServiceConfig("calld"));
}

SessionService::~SessionService() {
    YIO_LOG_INFO("SessionService destroy");
}

void SessionService::init() {
    auto user = dataStorage_->tryGetUser();
    auto xivaSecret = dataStorage_->tryGetXivaSecret();
    // always request xiva credentials for dedicated socket
    if (user.isNull() || settings_->params.useDedicatedXiva) {
        apiService_->requestUser(settings_->params.useDedicatedXiva);
    }
}

bool SessionService::isAuthorized() {
    const auto user = dataStorage_->tryGetUser();
    return !user.isNull() && !user["registration_status"].isNull();
}

bool SessionService::isConnected() {
    return connectionService_->isConnected();
}

bool SessionService::isTokenExpired() const {
    return tokenExpired_;
}

std::string SessionService::getOwnGuid() {
    const auto user = dataStorage_->tryGetUser();

    if (const auto guid = user["guid"]; guid.isString()) {
        return guid.asString();
    }

    return "";
}

bool SessionService::hasBgTasks() {
    return apiService_->hasTasks() || connectionService_->hasBgTasks() ||
           callingService_->hasBgTasks();
}

void SessionService::runOnBgIdle(std::function<void()> callback) {
    if (apiService_->hasTasks()) {
        apiService_->runOnIdle(std::move(callback));
    } else if (connectionService_->hasBgTasks()) {
        connectionService_->runOnBgIdle(std::move(callback));
    } else if (callingService_->hasBgTasks()) {
        callingService_->runOnBgIdle(std::move(callback));
    }
}

void SessionService::setAllowUsualCalls(bool allowUsualCalls) {
    callingService_->setAllowUsualCalls(allowUsualCalls);
}

void SessionService::sendHeartbeat() {
    connectionService_->sendHeartbeat();
}

std::shared_ptr<CallHolder> SessionService::getCallHolder() {
    return callingService_->getCallHolder();
}

void SessionService::startCall(const std::string& userGuid, const std::string& callPayload) {
    callingService_->startCall(userGuid, callPayload);
}

void SessionService::startCallToOwnDevice(const std::string& deviceId, const std::string& callPayload) {
    callingService_->startCallToOwnDevice(deviceId, callPayload);
}

void SessionService::declineIncomingCall() {
    return callingService_->declineIncomingCall();
}

void SessionService::acceptIncomingCall() {
    return callingService_->acceptIncomingCall();
}

void SessionService::hangupCall() {
    return callingService_->hangupCall();
}
