#include "data_storage.h"

#include <yandex_io/callkit/util/loop_thread.h>

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/folder/path.h>

#include <fstream>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

namespace {

    const char* persistentData_PATH_PREFIX =
        "/data/quasar/data/messenger/persistentData_v1_";

    bool createFilePath(const std::string& path) {
        try {
            TFsPath(path).Parent().MkDirs();
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("DataStorage.CreatePathFailed", "Cannot create file '" + path + "' what: " + e.what());
            return false;
        }
        return true;
    }

} // namespace

DataStorage::DataStorage(const std::string& storageId,
                         std::shared_ptr<LoopThread> workerThread)
    : workerThread_(std::move(workerThread))
{
    Y_VERIFY(workerThread_->checkInside());

    const std::string path = persistentData_PATH_PREFIX + storageId;
    if (createFilePath(path)) {
        filePath_ = path;
        std::ifstream fileStream(filePath_);
        if (fileStream.good()) {
            try {
                persistentData_ = quasar::parseJson(fileStream);
            } catch (const Json::Exception& e) {
                YIO_LOG_ERROR_EVENT("DataStorage.InvalidStorageJson", e.what());
                YIO_LOG_WARN("Removing malformed file " << path);
                save(); // clear malformed file
            }
        }
    }
}

DataStorage::~DataStorage() {
    Y_VERIFY(workerThread_->checkInside());
}

// {
//     "guid": "829e9e7b-38b9-40a5-a06a-9f9b92ae21f9",
//     "display_name": "just tea",
//     "registration_status": "U",
//     "version": 1582720757036522,
//     "avatar_id":
//     "user_avatar/yapic/64336/CVyMdAoTlGMxRNKcA3NfKq7PH0-1571116636",
//     "phone_id": "8de03053-7c42-4051-b556-686877665f74",
//     "uid": 673360205,
//     "phone": "+79266566248",
//     "contacts": []
// }
Json::Value DataStorage::tryGetUser() {
    return tryGetPersistentValue("messenger_user");
}

Json::Value DataStorage::tryGetXivaSecret() {
    return tryGetSessionValue("messenger_xiva_secret");
}

void DataStorage::setUser(const Json::Value& value) {
    setPersistentValue("messenger_user", value);
    userDataObservers_.notifyObservers();
}

void DataStorage::setXivaSecret(const Json::Value& value) {
    setSessionValue("messenger_xiva_secret", value);
    userDataObservers_.notifyObservers();
}

Json::Value DataStorage::tryGetPersistentValue(const char* key) {
    Y_VERIFY(workerThread_->checkInside());
    return persistentData_.get(key, Json::Value::null);
}

void DataStorage::setPersistentValue(const char* key,
                                     const Json::Value& value) {
    Y_VERIFY(workerThread_->checkInside());
    persistentData_[key] = value;
    shouldSave();
}

Json::Value DataStorage::tryGetSessionValue(const char* key) {
    Y_VERIFY(workerThread_->checkInside());
    return sessionData_.get(key, Json::Value::null);
}

void DataStorage::setSessionValue(const char* key, const Json::Value& value) {
    Y_VERIFY(workerThread_->checkInside());
    sessionData_[key] = value;
}

void DataStorage::shouldSave() {
    workerThread_->executeOnIdle([this] { save(); });
}

void DataStorage::save() {
    Y_VERIFY(workerThread_->checkInside());

    if (filePath_.empty()) {
        return;
    }
    std::ofstream fileStream(filePath_);
    if (fileStream.is_open()) {
        fileStream << quasar::jsonToString(persistentData_);
        fileStream.close();
    }
}

CallbackSubscription DataStorage::subscribeUserData(Callback callback) {
    return userDataObservers_.subscribe(callback);
}
