#include "async_service.h"

#include "weak_utils.h"

#include <yandex_io/libs/base/utils.h>

#include <util/system/yassert.h>

#include <algorithm>

using namespace messenger;

namespace {
    size_t MAX_REQUEST_COUNT = 4;
} // namespace

AsyncService::AsyncService(std::shared_ptr<LoopThread> workerThread)
    : workerThread_(std::move(workerThread))
{
}

AsyncService::~AsyncService() {
    Y_VERIFY(workerThread_->checkInside());
    while (!pendingTasks_.empty()) {
        auto request = pendingTasks_.front();
        request->setCancelled();
        pendingTasks_.pop();
    }
    for (auto request : activeTasks_) {
        request->setCancelled();
    }
    activeTasks_.clear();
    quasar::clear(idleCallbacks_);
}

void AsyncService::start(std::shared_ptr<AsyncTask> task) {
    Y_VERIFY(workerThread_->checkInside());
    if (task->isCancelled()) {
        return;
    }
    pendingTasks_.push(std::move(task));
    tryExecuteNext();
}

void AsyncService::cancel(std::shared_ptr<AsyncTask> task) {
    Y_VERIFY(workerThread_->checkInside());
    task->setCancelled();
}

void AsyncService::tryExecuteNext() {
    Y_VERIFY(workerThread_->checkInside());
    while (!pendingTasks_.empty() && activeTasks_.size() < MAX_REQUEST_COUNT) {
        auto task = pendingTasks_.front();
        pendingTasks_.pop();
        if (task->isCancelled()) {
            continue;
        }
        activeTasks_.push_back(task);
        std::function<void()> callback = [this, task]() mutable {
            onExecuted(std::move(task));
        };
        std::thread(&AsyncService::executeInBg, weak_from(workerThread_),
                    std::move(task), std::move(callback))
            .detach();
    }
    processIdleState();
}

void AsyncService::processIdleState() {
    if (!hasTasks()) {
        while (!idleCallbacks_.empty()) {
            auto callback = idleCallbacks_.front();
            idleCallbacks_.pop();
            if (callback) {
                callback();
            }
        }
    }
}

// static
void AsyncService::executeInBg(std::weak_ptr<LoopThread> workerThread,
                               std::shared_ptr<AsyncTask> task,
                               std::function<void()> callback) {
    if (workerThread.expired()) {
        return;
    }
    if (!task->isCancelled()) {
        task->run();
    }
    if (auto loop = workerThread.lock()) {
        Y_VERIFY(loop->checkOutside());
        loop->execute(std::move(callback));
    }
}

void AsyncService::onExecuted(std::shared_ptr<AsyncTask> task) {
    Y_VERIFY(workerThread_->checkInside());
    if (!task->isCancelled()) {
        task->onFinished();
    }
    activeTasks_.erase(
        std::remove(activeTasks_.begin(), activeTasks_.end(), task),
        activeTasks_.end());
    tryExecuteNext();
}

bool AsyncService::hasTasks() const {
    Y_VERIFY(workerThread_->checkInside());
    return !pendingTasks_.empty() || !activeTasks_.empty();
}

void AsyncService::runOnIdle(std::function<void()> callback) {
    Y_VERIFY(workerThread_->checkInside());
    idleCallbacks_.push(std::move(callback));
    processIdleState();
}
