#include "loop_thread.h"

#include "weak_utils.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger;

class messenger::Executor
    : public std::enable_shared_from_this<messenger::Executor> {
public:
    // on: worker (related looper) thread
    Executor(std::weak_ptr<LoopThread> loopThread);
    virtual ~Executor();

    // callbacks won't be run after Executor destroyed
    // on: worker thread
    void execute(const std::function<void()>& task);
    // on: worker thread
    void executeOnIdle(const std::function<void()>& task);
    // on: worker thread
    void executeDelayed(const std::function<void()>& task,
                        std::chrono::milliseconds delay);

private:
    std::weak_ptr<LoopThread> loopThread_;
};

// static
std::shared_ptr<LoopThread> LoopThread::create() {
    return std::shared_ptr<LoopThread>(new LoopThread());
}

LoopThread::LoopThread()
    : queue_(1024,
             [](size_t /*size*/) {
                 Y_VERIFY(false);
                 YIO_LOG_ERROR_EVENT("LoopThread.TaskOverflow", "Task queue overflow!!");
             })
    , threadId_()
    , inited_(false)
{
    queueThread_ = std::thread(&LoopThread::run, this);
}

LoopThread::~LoopThread() {
    destroyBlocked();
}

void LoopThread::execute(std::function<void()> task) {
    queue_.pushTask(task);
}

void LoopThread::executeOnIdle(std::function<void()> task) {
    queue_.pushIdleTask(task);
}

void LoopThread::executeDelayed(std::function<void()> task,
                                std::chrono::milliseconds delay) {
    queue_.pushDelayedTask(task, delay);
}

bool LoopThread::isIdle() {
    return queue_.isIdle();
}

void LoopThread::destroyBlocked() {
    if (!queue_.shutdown()) {
        return;
    }
    if (queueThread_.joinable()) {
        queueThread_.join();
    }
}

void LoopThread::destroy() {
    queue_.shutdown();
}

void LoopThread::join() {
    if (queueThread_.joinable()) {
        queueThread_.join();
    }
}

void LoopThread::run() {
    init();
    std::function<void()> task;
    do {
        queue_.popTask(task); // sleeps until there is a callback
        if (task) {
            runOnce(task);
        }
    } while (task);
    // Tasks can hold the refs which should be disposed on the worker thread.
    queue_.clear();
}

void LoopThread::runOnce(const std::function<void()>& callback) {
    callback();
}

// static
ScopedExecutor ScopedExecutor::create(std::shared_ptr<LoopThread> loopThread) {
    auto executor = std::make_shared<Executor>(loopThread);
    return ScopedExecutor(std::move(executor));
}

// static
ScopedExecutor ScopedExecutor::create(LoopThread* loopThread) {
    auto executor = std::make_shared<Executor>(weak_from(loopThread));
    return ScopedExecutor(std::move(executor));
}

bool LoopThread::checkInside() const {
    Y_VERIFY(inited_);
    auto current = std::this_thread::get_id();
    if (threadId_ == current) {
        return true;
    }
    YIO_LOG_ERROR_EVENT("LoopThread.CheckInsideFailed", "Thread " << current << " must be " << threadId_);
    return false;
}

bool LoopThread::checkOutside() const {
    Y_VERIFY(inited_);
    auto current = std::this_thread::get_id();
    if (threadId_ != current) {
        return true;
    }
    YIO_LOG_ERROR_EVENT("LoopThread.CheckOutsideFailed", "Thread " << current << " must not be " << threadId_);
    return false;
}

void LoopThread::init() {
    Y_VERIFY(!inited_ && "This check inited twice");
    threadId_.exchange(std::this_thread::get_id());
    inited_.exchange(true);
}

ScopedExecutor::operator bool() const {
    return executor_ != nullptr;
}

void ScopedExecutor::reset() {
    executor_.reset();
}

ScopedExecutor::ScopedExecutor(std::shared_ptr<Executor> looper)
    : executor_(looper)
{
}

void ScopedExecutor::execute(const std::function<void()>& task) {
    if (!executor_) {
        throw std::runtime_error("Trying to execute on cancelled executor");
    }
    executor_->execute(task);
}

void ScopedExecutor::executeOnIdle(const std::function<void()>& task) {
    if (!executor_) {
        throw std::runtime_error("Trying to execute on cancelled executor");
    }
    executor_->executeOnIdle(task);
}

void ScopedExecutor::executeDelayed(const std::function<void()>& task,
                                    std::chrono::milliseconds delay) {
    if (!executor_) {
        throw std::runtime_error("Trying to execute on cancelled executor");
    }
    executor_->executeDelayed(task, delay);
}

Executor::Executor(std::weak_ptr<LoopThread> loopThread)
    : loopThread_(loopThread)
{
}

Executor::~Executor() {
    // Parent object must live on the same looper
    auto looper = loopThread_.lock();
    if (looper) {
        looper->checkInside();
    }
}

void Executor::execute(const std::function<void()>& task) {
    auto looper = loopThread_.lock();
    if (looper) {
        looper->checkInside();
        auto weak = weak_from(this);
        looper->execute([weak, task] {
            if (!weak.expired()) {
                task();
            }
        });
    }
}

void Executor::executeOnIdle(const std::function<void()>& task) {
    auto looper = loopThread_.lock();
    if (looper) {
        looper->checkInside();
        auto weak = weak_from(this);
        looper->executeOnIdle([weak, task] {
            if (!weak.expired()) {
                task();
            }
        });
    }
}

void Executor::executeDelayed(const std::function<void()>& task,
                              std::chrono::milliseconds delay) {
    auto looper = loopThread_.lock();
    if (looper) {
        looper->checkInside();
        auto weak = weak_from(this);
        looper->executeDelayed(
            [weak, task] {
                if (!weak.expired()) {
                    task();
                }
            },
            delay);
    }
}
