#pragma once

#include "task_queue.h"

#include <atomic>
#include <chrono>
#include <memory>
#include <thread>

namespace messenger {

    class Executor;
    class LoopThread;

    // The cancellable executor, useful for timers.
    //
    // on: worker thread
    class ScopedExecutor {
    public:
        static ScopedExecutor create(std::shared_ptr<LoopThread> loopThread);
        static ScopedExecutor create(LoopThread* loopThread);
        ScopedExecutor() = default;
        virtual ~ScopedExecutor() = default;
        ScopedExecutor(ScopedExecutor&& other) = default;
        ScopedExecutor& operator=(ScopedExecutor&&) = default;

        // callbacks won't be run after Executor destroyed
        // on: worker thread
        void execute(const std::function<void()>& task);
        // on: worker thread
        void executeOnIdle(const std::function<void()>& task);
        // on: worker thread
        void executeDelayed(const std::function<void()>& task,
                            std::chrono::milliseconds delay);

        explicit operator bool() const;
        void reset();

    private:
        friend class LoopThread;

        ScopedExecutor(std::shared_ptr<Executor> looper);

        ScopedExecutor& operator=(const ScopedExecutor&) = delete;
        ScopedExecutor(const ScopedExecutor&) = delete;

        std::shared_ptr<Executor> executor_;
    };

    class LoopThread: public std::enable_shared_from_this<LoopThread> {
    public:
        static std::shared_ptr<LoopThread> create();

        virtual ~LoopThread();

        void destroyBlocked();
        void destroy();
        void join();
        bool isIdle();

        void execute(std::function<void()> task);
        void executeOnIdle(std::function<void()> task);
        void executeDelayed(std::function<void()> task,
                            std::chrono::milliseconds delay);

        bool checkInside() const;
        bool checkOutside() const;

    private:
        friend class TaskRunner;

        LoopThread();

        void run();
        static void runOnce(const std::function<void()>& callback);

        void init();

        TaskQueue<std::function<void()>> queue_;
        std::thread queueThread_;
        std::atomic<std::thread::id> threadId_;
        std::atomic_bool inited_;
    };

} // namespace messenger
