#pragma once

#include "weak_utils.h"

#include <algorithm>
#include <functional>
#include <memory>
#include <mutex>
#include <type_traits>
#include <vector>

namespace messenger {

    // ObserverType should be callable by parentheses and assignable with equal
    // sign.
    // Observers must not do long work when notified, their calls are synchronized.
    // Observers can add or destroy subscriptions when notified.
    //
    // Common use is:
    //
    // class Producer {
    // public:
    //    using Callback = std::function<void(int)>;
    //    using Subscription = ObserverList<int>::Subscription;
    //
    //    std::unique_ptr<Subscription> subscribe(Callback callback) {
    //        return observerList_.subscribe(callback);
    //    }
    //
    //    void onReceiveNumber(int value) {
    //        observerList_.notifyObservers(value);
    //    }
    //
    // private:
    //    ObserverList<int> observerList_;
    //};
    //
    // class Consumer {
    // public:
    //    Consumer(Producer& producer) {
    //        subscription_ = producer.subscribe(
    //            std::bind(&Consumer::numberArrived, this, std::placeholders::_1));
    //    }
    //
    //    virtual ~Consumer() {
    //        subscription_.reset();
    //        // no callback will be run after subscription is destroyed
    //    }
    //
    // private:
    //    void numberArrived(int value) { std::cout << value; }
    //
    //    std::unique_ptr<Producer::Subscription> subscription_;
    //};
    //
    // on: any thread
    template <typename... ObserverArgs>
    class ObserverList {
        struct ObserverHolder;

    public:
        // used only with unique_ptr<>
        class Subscription {
        public:
            virtual ~Subscription() {
                cancel();
            }

        private:
            friend class ObserverList;

            void cancel() {
                if (auto holder = observerHolder_.lock()) {
                    holder->cancelSubscription();
                }
            }

            Subscription(std::weak_ptr<ObserverHolder> observerHolder)
                : observerHolder_(observerHolder)
            {
            }

            std::weak_ptr<ObserverHolder> observerHolder_;
        };

        ObserverList()
            : mutex_(new std::recursive_mutex())
        {
        }

        virtual ~ObserverList() {
            std::lock_guard<std::recursive_mutex> guard(*mutex_);
            // WeakPtr<ObserverList> for the poor
            for (auto& holder : observers_) {
                holder->parentPtr = nullptr;
            }
            observers_.clear();
        };

        bool empty() {
            std::lock_guard<std::recursive_mutex> guard(*mutex_);
            return observers_.empty();
        }

        using ObserverType = std::function<void(ObserverArgs...)>;
        using ScopedSubscription = std::unique_ptr<Subscription>;

        [[nodiscard]] ScopedSubscription subscribe(ObserverType observer) {
            std::lock_guard<std::recursive_mutex> guard(*mutex_);
            auto observerHolder =
                std::make_shared<ObserverHolder>(this, std::move(observer), mutex_);
            observers_.push_back(observerHolder);
            return std::unique_ptr<Subscription>(new Subscription(observerHolder));
        }

        void notifyObservers(ObserverArgs... args) {
            std::lock_guard<std::recursive_mutex> guard(*mutex_);
            std::vector<std::shared_ptr<ObserverHolder>> observersCopy = observers_;
            for (const auto& observerHolder : observersCopy) {
                if (std::find(observers_.begin(), observers_.end(),
                              observerHolder) !=
                    observers_.end()) { // may be deleted in callback
                    observerHolder->observer(args...);
                }
            }
        }

        template <class T>
        [[nodiscard]] ScopedSubscription subscribe(T weakablePtr,
                                                   ObserverType observer) {
            auto weak = weak_from(weakablePtr);
            ObserverType observerWrapper = [observer, weak](ObserverArgs... args) {
                if (auto p = weak.lock()) {
                    observer(args...);
                }
            };
            return subscribe(observerWrapper);
        }

        template <class T, class V>
        [[nodiscard]] ScopedSubscription
        subscribe(T weakablePtr1, V weakablePtr2, ObserverType observer) {
            auto weak1 = weak_from(weakablePtr1);
            auto weak2 = weak_from(weakablePtr2);
            ObserverType observerWrapper = [observer, weak1,
                                            weak2](ObserverArgs... args) {
                if (auto p1 = weak1.lock()) {
                    if (auto p2 = weak2.lock()) {
                        observer(args...);
                    }
                }
            };
            return subscribe(observerWrapper);
        }

    private:
        ObserverList(const ObserverList&) = delete;
        ObserverList& operator=(const ObserverList&) = delete;

        struct ObserverHolder {
            ObserverHolder(ObserverList<ObserverArgs...>* parentPtr,
                           ObserverType observer,
                           std::shared_ptr<std::recursive_mutex> mutex)
                : parentPtr(parentPtr)
                , observer(std::move(observer))
                , mutex(std::move(mutex))
            {
            }

            void cancelSubscription() {
                // WeakPtr<ObserverList> for the poor
                std::lock_guard<std::recursive_mutex> guard(*mutex);
                if (!parentPtr) {
                    return;
                }
                parentPtr->cancelSubscription(this);
                parentPtr = nullptr;
            }

            ObserverList<ObserverArgs...>* parentPtr;
            const ObserverType observer;
            std::shared_ptr<std::recursive_mutex> mutex;
        };

        void cancelSubscription(const ObserverHolder* holder) {
            std::lock_guard<std::recursive_mutex> guard(*mutex_);
            for (auto it = observers_.begin(); it != observers_.end(); it++) {
                if ((*it).get() == holder) {
                    observers_.erase(it);
                    break;
                }
            }
        }
        // subscribers can modify the list while iterating
        std::shared_ptr<std::recursive_mutex> mutex_;
        std::vector<std::shared_ptr<ObserverHolder>> observers_;
    };

    // Simple implementation - just to write less code
    using Callback = std::function<void(void)>;
    using CallbackObserverList = ObserverList<>;
    using CallbackSubscription = CallbackObserverList::ScopedSubscription;

} // namespace messenger
