#pragma once

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <chrono>
#include <functional>
#include <mutex>
#include <queue>

namespace messenger {

    // The version of quasar::TaskQueue for LoopThread that holds a regular task
    // queue and a  queue for idle tasks.
    template <typename T>
    class TaskQueue {
        using DelayedTask =
            std::pair<T, std::chrono::time_point<std::chrono::steady_clock>>;

    public:
        using Callback = std::function<void(size_t)>;

        TaskQueue(int maxSize, Callback onOverflow = Callback(),
                  Callback onBecomeNormal = Callback())
            : finished_(false)
            , maxSize_(maxSize)
            , onOverflow_(std::move(onOverflow))
            , onBecomeNormal_(std::move(onBecomeNormal))
        {
        }

        void popTask(T& task) {
            std::unique_lock<std::mutex> lock(mutex_);
            bool hasTask = false;
            do {
                waitTaskLocked(lock);
                hasTask = popTaskLocked(task);
            } while (!hasTask);

            const int newSize =
                queue_.size() + idleQueue_.size() + delayedQueue_.size();
            lock.unlock();

            // become normal on maxSize - 1 to reduce oscillation processes
            if (newSize == maxSize_ - 1 && onBecomeNormal_) {
                onBecomeNormal_(newSize);
            }
        }

        void pushTask(T task) {
            std::unique_lock<std::mutex> lock(mutex_);
            if (finished_) {
                return;
            }
            const int oldSize =
                queue_.size() + idleQueue_.size() + delayedQueue_.size();
            queue_.push(std::move(task));
            lock.unlock();
            cond_.notify_one();
            if (oldSize == maxSize_ && onOverflow_) {
                onOverflow_(oldSize + 1);
            }
        }

        void pushIdleTask(T task) {
            std::unique_lock<std::mutex> lock(mutex_);
            if (finished_) {
                return;
            }
            const int oldSize =
                queue_.size() + idleQueue_.size() + delayedQueue_.size();
            idleQueue_.push(std::move(task));
            lock.unlock();
            cond_.notify_one();
            if (oldSize == maxSize_ && onOverflow_) {
                onOverflow_(oldSize + 1);
            }
        }

        void pushDelayedTask(T task, std::chrono::milliseconds delay) {
            std::unique_lock<std::mutex> lock(mutex_);
            if (finished_) {
                return;
            }
            const int oldSize =
                queue_.size() + idleQueue_.size() + delayedQueue_.size();
            delayedQueue_.push(
                DelayedTask(task, std::chrono::steady_clock::now() + delay));
            lock.unlock();
            cond_.notify_one();
            if (oldSize == maxSize_ && onOverflow_) {
                onOverflow_(oldSize + 1);
            }
        }

        bool isEmpty() const {
            std::lock_guard<std::mutex> lock(mutex_);
            return queue_.empty() && idleQueue_.empty() && delayedQueue_.empty();
        }

        bool isIdle() const {
            std::lock_guard<std::mutex> lock(mutex_);
            return queue_.empty();
        }

        void clear() {
            std::lock_guard<std::mutex> lock(mutex_);
            quasar::clear(queue_);
            quasar::clear(idleQueue_);
            quasar::clear(delayedQueue_);
        }

        bool shutdown() {
            std::unique_lock<std::mutex> lock(mutex_);
            if (finished_) {
                return false;
            }
            finished_ = true;
            queue_.push(std::move(T()));
            quasar::clear(idleQueue_);
            quasar::clear(delayedQueue_);
            cond_.notify_one();
            return true;
        }

    private:
        TaskQueue(const TaskQueue&) = delete;            // disable copying
        TaskQueue& operator=(const TaskQueue&) = delete; // disable assignment

        void waitTaskLocked(std::unique_lock<std::mutex>& lock) {
            if (delayedQueue_.empty()) {
                auto condition = [this] {
                    return !queue_.empty() || !idleQueue_.empty() ||
                           !delayedQueue_.empty();
                };
                cond_.wait(lock, condition);
            } else {
                auto until = delayedQueue_.top().second;
                auto condition = [this, until] {
                    return !queue_.empty() || !idleQueue_.empty() ||
                           delayedQueue_.empty() ||
                           (!delayedQueue_.empty() &&
                            (delayedQueue_.top().second <
                                 std::chrono::steady_clock::now() ||
                             delayedQueue_.top().second < until));
                };
                cond_.wait_until(lock, until, condition);
            }
        }

        bool popTaskLocked(T& task) {
            if (!delayedQueue_.empty() &&
                delayedQueue_.top().second < std::chrono::steady_clock::now()) {
                task = std::move(delayedQueue_.top().first);
                delayedQueue_.pop();
                return true;
            }
            if (!queue_.empty()) {
                task = std::move(queue_.front());
                queue_.pop();
                return true;
            }
            if (!idleQueue_.empty()) {
                task = std::move(idleQueue_.front());
                idleQueue_.pop();
                return true;
            }
            return false;
        }

        struct DelayedTaskCmp {
            bool operator()(const DelayedTask& l, const DelayedTask& r) {
                return l.second > r.second;
            }
        };
        std::queue<T> queue_;
        std::queue<T> idleQueue_;
        std::priority_queue<DelayedTask, std::vector<DelayedTask>, DelayedTaskCmp>
            delayedQueue_;
        mutable std::mutex mutex_;
        quasar::SteadyConditionVariable cond_;
        bool finished_;

        int maxSize_;
        const Callback onOverflow_;
        const Callback onBecomeNormal_;
    };

} // namespace messenger
