#pragma once

#include <functional>
#include <memory>
#include <type_traits>

namespace messenger {

#define CREATE_FROM_INIT(ClassName)                                            \
public:                                                                        \
    template <typename... Args>                                                \
    static std::shared_ptr<ClassName> create(Args... args) {                   \
        static_assert(std::is_base_of<std::enable_shared_from_this<ClassName>, \
                                      ClassName>::value);                      \
        std::shared_ptr<ClassName> ptr =                                       \
            std::shared_ptr<ClassName>(new ClassName());                       \
        ptr->init(args...);                                                    \
        return ptr;                                                            \
    }

    template <class T>
    std::weak_ptr<T> weak_from(T* ptr) {
        static_assert(std::is_base_of<std::enable_shared_from_this<T>, T>::value);
        return std::weak_ptr<T>(ptr->shared_from_this());
    }

    template <class T>
    std::weak_ptr<T> weak_from(std::shared_ptr<T> shared) {
        return std::weak_ptr<T>(shared);
    }

    template <class T>
    std::weak_ptr<T> weak_from(std::weak_ptr<T> weak) {
        return std::move(weak);
    }

    // func is called with locked weak ptr
    // T, V - std::weak_ptr, std::shared_ptr or std::enable_shared_from_this*
    template <class T>
    std::function<void()> bindWeak(T weakablePtr, std::function<void()> func) {
        auto weak = weak_from(weakablePtr);
        return [weak, func] {
            if (auto p = weak.lock()) {
                func();
            }
        };
    }

    // func is called with locked weak ptr
    // T, V - std::weak_ptr, std::shared_ptr or std::enable_shared_from_this*
    template <class T, class V>
    std::function<void()> bindWeak(T weakablePtr1, V weakablePtr2,
                                   std::function<void()> func) {
        auto weak1 = weak_from(weakablePtr1);
        auto weak2 = weak_from(weakablePtr2);
        return [weak1, weak2, func] {
            if (auto p1 = weak1.lock()) {
                if (auto p2 = weak2.lock()) {
                    func();
                }
            }
        };
    }

    template <class T>
    class WeakRef {
        static_assert(std::is_base_of<std::enable_shared_from_this<T>, T>::value);

    public:
        WeakRef() = default;
        void init(T* thisPtr) {
            Y_VERIFY(!ptr_);
            ptr_ = thisPtr->shared_from_this();
        }
        std::function<void()> bind(std::function<void(T*)> func) const {
            auto weakRef = *this;
            return [weakRef, func]() {
                if (weakRef) {
                    func(weakRef.get());
                }
            };
        }

        std::function<void()> bind(std::function<void()> func) const {
            auto weakRef = *this;
            return [weakRef, func]() {
                if (weakRef) {
                    func();
                }
            };
        }
        WeakRef& operator=(const WeakRef&) = default;
        ~WeakRef() = default;

        T* get() const {
            return ptr_->pointer;
        }

        T& operator*() const {
            Y_VERIFY(get() != nullptr);
            return *get();
        }

        T* operator->() const {
            Y_VERIFY(get() != nullptr);
            return get();
        }

        explicit operator bool() const {
            return get() != nullptr;
        }

    private:
        const std::weak_ptr<T> ptr_;
    };

} // namespace messenger
