#pragma once

#include <contrib/libs/msgpack/include/msgpack.hpp>

#include <cstdint>
#include <string>
#include <vector>

namespace messenger {
    namespace xiva {

        // See also
        // https://a.yandex-team.ru/arc/trunk/arcadia/mail/xiva/core/include/yxiva/core/binary_protocol.h
        enum FrameType: std::uint8_t {
            DATA = 1,         // data frame from client to rproxy backends
            PROXY_STATUS = 2, // xiva response if rproxy backend is unavailable
            PUSH = 3,         // push notifications from any authorized sender, see PushHeader

            MAX,
        };

        struct DataHeader {
            std::uint8_t serviceId = 0;
            std::uint32_t requestId = 0;
            std::string path;

            MSGPACK_DEFINE(serviceId, requestId, path);
        };

        struct ProxyStatusHeader {
            std::uint32_t requestId = 0;
            std::uint16_t errorCode = 0;

            MSGPACK_DEFINE(requestId, errorCode);
        };

        struct PushHeader {
            std::string uid;
            std::string service;
            std::string event;
            std::string transitId;

            MSGPACK_DEFINE(uid, service, event, transitId);
        };

        enum ErrorCode: std::uint16_t {
            SUCCESS = 0,
            PROTOCOL_ERROR = 1,
            BACKEND_CALL_ERROR = 2,
            INTERNAL_ERROR = 3,
            CORRUPTED_DATA_HEADER = 4,
            BACKEND_NOT_FOUND = 5,
        };

        static const std::vector<std::string> ERROR_NAMES = {
            "SUCCESS",
            "PROTOCOL_ERROR",
            "BACKEND_CALL_ERROR",
            "INTERNAL_ERROR",
            "CORRUPTED_DATA_HEADER",
            "BACKEND_NOT_FOUND",
        };

        inline std::string errorMessage(std::uint16_t errorCode) {
            return errorCode > ERROR_NAMES.size() ? "UNKNOWN" : ERROR_NAMES[errorCode];
        }

    } // namespace xiva
} // namespace messenger
