#pragma once

#include <yandex_io/callkit/util/observer_list.h>

#include <cstdint>
#include <functional>
#include <mutex>
#include <string>
#include <vector>

namespace messenger {
    namespace xiva {

        // The interface of an abstract xiva socket that can handle multiple services.
        // We need this interface to make two implementation - explicit web socket that
        // runs under our control and a socket living in pushd daemon.
        //
        // As pushes are implemented inside the new binary protocol, no ordinal pushes
        // are considered here.
        //
        // on: any thread
        class SocketApi {
        public:
            enum class Status {
                DISCONNECTED,
                CONNECTING,
                CONNECTED,
            };

            using DataObserver = std::function<void(const std::string& /*buffer*/)>;
            using DataSubscription =
                ObserverList<const std::string&>::ScopedSubscription;
            using StatusObserver = std::function<void(Status)>;
            using StatusSubscription = ObserverList<Status>::ScopedSubscription;
            using FailCallback = std::function<void()>;

            SocketApi();
            virtual ~SocketApi() = default;

            virtual std::vector<std::string> getServiceNames() = 0;

            // Callback calls are not guaranteed.
            // It is recommended for the implementation to handle possible errors.
            // If called, called on bg thread.
            virtual void sendBinaryData(const std::string& buffer,
                                        FailCallback onFail = FailCallback()) = 0;
            virtual void sendTextData(const std::string& buffer,
                                      FailCallback onFail = FailCallback()) = 0;
            Status getStatus();
            [[nodiscard]] DataSubscription subscribeBinaryData(DataObserver observer);
            [[nodiscard]] DataSubscription subscribeTextData(DataObserver observer);
            [[nodiscard]] StatusSubscription subscribe(StatusObserver observer);

        protected:
            void handleBinaryData(const std::string& data);
            void handleTextData(const std::string& text);
            void handleStatus(Status status);

        private:
            ObserverList<const std::string&> binaryObservers_;
            ObserverList<const std::string&> textObservers_;
            ObserverList<Status> statusObservers_;

            Status currentStatus_ = Status::DISCONNECTED;
            StatusSubscription selfStatusSubscription_;
        };

    } // namespace xiva
} // namespace messenger
