#include "transport.h"

#include "protocol.h"
#include "request.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

#include <utility>

YIO_DEFINE_LOG_MODULE("callkit");

using namespace messenger::xiva;

Transport::Transport(std::shared_ptr<SocketApi> socketApi,
                     std::string serviceName)
    : socketApi_(std::move(socketApi))
    , serviceName_(std::move(serviceName))
    , nextRequestId_(1)
    , destroyed_(false)
    , requestTimeout_(5000)
{
    auto services = socketApi_->getServiceNames();
    auto it = std::find(services.begin(), services.end(), serviceName_);
    Y_VERIFY(it != services.end());
    if (it == services.end()) {
        YIO_LOG_ERROR_EVENT("XivaTransport.NoMessageServicesInXivaSubscribtions", "Xiva connection does not setup messenger service");
        serviceIndex_ = 0xFF;
        return;
    }
    serviceIndex_ = std::distance(services.begin(), it);
    dataSubscription_ = socketApi_->subscribeBinaryData(
        std::bind(&Transport::onReceive, this, std::placeholders::_1));
    statusSubscription_ = socketApi_->subscribe(
        std::bind(&Transport::onSocketStatus, this, std::placeholders::_1));
}

Transport::~Transport() {
    Y_VERIFY(!destroyed_);
    destroyed_ = true;
    cancelAll();
    timerQueue_.destroy();
}

uint32_t Transport::send(const std::shared_ptr<Request>& request) {
    if (destroyed_) {
        return 0u;
    }
    auto requestId = startRequest(request);
    if (requestId == 0u) {
        return 0u;
    }

    std::stringstream buffer;

    buffer.put(FrameType::DATA);

    DataHeader header;
    header.serviceId = serviceIndex_;
    header.requestId = requestId;
    header.path = request->getPath();
    msgpack::pack(buffer, header);

    buffer << request->getPayload();
    auto data = buffer.str();
    std::weak_ptr<Transport> weakme(shared_from_this());
    socketApi_->sendBinaryData(data, [weakme, requestId] {
        auto me = weakme.lock();
        if (me) {
            me->handleMessageSendFail(requestId);
        }
    });
    YIO_LOG_DEBUG("Xiva dataframe sent: " + std::to_string(header.serviceId) + "/" +
                  header.path +
                  ", data:" + quasar::base64Encode(data.c_str(), data.size()));
    timerQueue_.addDelayed([this, requestId] { handleTimeout(requestId); },
                           requestTimeout_);
    return requestId;
}

void Transport::cancel(uint32_t requestId) {
    if (destroyed_) {
        return;
    }
    auto request = finishRequest(requestId);
    if (!request) {
        return;
    }
    request->onCancel();
}

void Transport::cancelAll() {
    std::lock_guard<std::mutex> lock(mutex_);
    auto requestsCopy = activeRequests_;
    for (auto& itCopy : requestsCopy) {
        auto it = activeRequests_.find(itCopy.first);
        if (it != activeRequests_.end()) {
            it->second->onCancel();
            activeRequests_.erase(it);
        }
    }
}

Transport::PushSubscription Transport::subscribeOnPush(PushObserver observer) {
    return pushObservers_.subscribe(observer);
}

void Transport::onReceive(const std::string& frameData) {
    if (destroyed_) {
        return;
    }
    Y_VERIFY(frameData.size());
    if (frameData.empty()) {
        YIO_LOG_ERROR_EVENT("XivaTransport.ReceivedEmptyFrame", "onReceive: empty frame");
        // No way to detect a request id, just skip
        return;
    }

    std::size_t offset = 0u;

    char type = frameData[0];
    offset++;
    try {
        switch (type) {
            case FrameType::DATA:
                handleDataFrame(frameData, &offset);
                break;
            case FrameType::PROXY_STATUS:
                handleProxyStatus(frameData, &offset);
                break;
            case FrameType::PUSH:
                handlePush(frameData, &offset);
                break;
            default:
                YIO_LOG_ERROR_EVENT("XivaTransport.InvalidFrameType", "onReceive: unknown type " + std::to_string(type));
                Y_VERIFY(!"Not implemented");
                return;
        }
    } catch (const msgpack::type_error& e) {
        // No way to detect a request, just skip
        YIO_LOG_ERROR_EVENT("XivaTransport.HandleFrame.MsgpackException", e.what());
        return;
    } catch (const std::bad_cast& e) {
        YIO_LOG_ERROR_EVENT("XivaTransport.HandleFrame.BadCastException", e.what());
        return;
    }
}

void Transport::onSocketStatus(SocketApi::Status status) const {
    if (destroyed_) {
        return;
    }
    // No idea whether we need to handle the connection status here
}

void Transport::handleMessageSendFail(uint32_t requestId) {
    if (destroyed_) {
        return;
    }
    auto request = finishRequest(requestId);
    if (!request) {
        return;
    }
    request->onFail();
}

void Transport::handleTimeout(uint32_t requestId) {
    if (destroyed_) {
        return;
    }
    auto request = finishRequest(requestId);
    if (!request) {
        return;
    }
    request->onTimeout();
}

uint32_t Transport::startRequest(std::shared_ptr<Request> request) {
    if (destroyed_ || serviceIndex_ == 0xFF) {
        return 0u;
    }
    std::lock_guard<std::mutex> lock(mutex_);
    Y_VERIFY(activeRequests_.count(nextRequestId_) == 0);
    for (auto& entry : activeRequests_) {
        if (entry.second == request) {
            YIO_LOG_ERROR_EVENT("XivaTransport.RequestAlreadySent", "Request was already sent: " + request->getPath());
            Y_VERIFY(!"Request was already sent");
            return 0u;
        }
    }
    activeRequests_[nextRequestId_] = request;
    return nextRequestId_++;
}

std::shared_ptr<Request> Transport::finishRequest(uint32_t requestId) {
    std::lock_guard<std::mutex> lock(mutex_);
    auto it = activeRequests_.find(requestId);
    if (it == activeRequests_.end()) {
        return std::shared_ptr<Request>();
    }
    auto request = it->second;
    activeRequests_.erase(it);
    return request;
}

void Transport::handleDataFrame(const std::string& frameData,
                                std::size_t* offset) {
    DataHeader header;
    msgpack::unpack(frameData.data(), frameData.size(), *offset)
        .get()
        .convert(header);
    if (serviceIndex_ != header.serviceId) {
        return;
    }
    auto request = finishRequest(header.requestId);
    if (!request) { // may be cancelled
        YIO_LOG_WARN("handleDataFrame: no active request with id=" +
                     std::to_string(header.requestId));
        return;
    }
    Y_VERIFY(header.path == request->getPath());
    auto payload = frameData.substr(*offset);
    *offset += payload.size();
    YIO_LOG_DEBUG("Xiva dataframe received: " +
                  quasar::base64Encode(payload.c_str(), payload.size()));
    request->onDataFrame(payload);
}

void Transport::handleProxyStatus(const std::string& frameData,
                                  std::size_t* offset) {
    ProxyStatusHeader header;
    msgpack::unpack(frameData.data(), frameData.size(), *offset)
        .get()
        .convert(header);
    auto request = finishRequest(header.requestId);
    if (!request) { // may be cancelled
        YIO_LOG_WARN("handleDataFrame: no active request with id="
                     << header.requestId << " and error=" << header.errorCode
                     << " (" << errorMessage(header.errorCode));
        return;
    }
    auto err = header.errorCode;
    std::string errName =
        err < ERROR_NAMES.size() ? ERROR_NAMES[err] : "Unknown";
    YIO_LOG_DEBUG("Xiva proxystatus received: " + errName + " (" +
                  std::to_string(err) + ")");
    request->onProxyStatus(err, errName);
}

void Transport::handlePush(const std::string& frameData, std::size_t* offset) {
    PushHeader header;
    msgpack::unpack(frameData.data(), frameData.size(), *offset)
        .get()
        .convert(header);
    if (header.service != serviceName_ || header.event != "delivery") {
        return;
    }

    auto payload = frameData.substr(*offset);
    *offset += payload.size();

    YIO_LOG_DEBUG("Xiva push received: " << payload.size());

    pushObservers_.notifyObservers(payload);
}
