#pragma once

#include "socket_api.h"

#include <yandex_io/callkit/util/observer_list.h>

#include <yandex_io/libs/base/named_callback_queue.h>

#include <chrono>
#include <map>
#include <memory>
#include <mutex>
#include <string>

namespace messenger {
    namespace xiva {

        class Request;

        // This transport DOES NOT creates its own socket sending it to pushd.
        // See https://wiki.yandex-team.ru/yxiva/spec/drafts/rproxy/ and
        // mail/xiva/core/include/yxiva/core/binary_protocol.h for protocol
        //
        // on: any thread
        // all work (serialization) is done on new threads
        class Transport: public std::enable_shared_from_this<Transport> {
        public:
            Transport(std::shared_ptr<SocketApi> socketApi, std::string serviceName);
            virtual ~Transport();
            uint32_t send(const std::shared_ptr<Request>& request);
            void cancel(uint32_t requestId);
            void cancelAll();

            using PushSubscription =
                ObserverList<const std::string&>::ScopedSubscription;
            using PushObserver = ObserverList<const std::string&>::ObserverType;
            [[nodiscard]] PushSubscription subscribeOnPush(PushObserver observer);

        private:
            void onReceive(const std::string& frameData);
            void onSocketStatus(SocketApi::Status status) const;

            void handleTimeout(uint32_t requestId);

            uint32_t startRequest(std::shared_ptr<Request> request);
            std::shared_ptr<Request> finishRequest(uint32_t requestId);

            void handleMessageSendFail(uint32_t requestId);
            void handleDataFrame(const std::string& frameData, std::size_t* offset);
            void handleProxyStatus(const std::string& frameData, std::size_t* offset);
            void handlePush(const std::string& frameData, std::size_t* offset);

            std::shared_ptr<SocketApi> socketApi_;
            std::string serviceName_;
            uint8_t serviceIndex_;

            std::map<unsigned, std::shared_ptr<Request>> activeRequests_;
            uint32_t nextRequestId_;
            std::mutex mutex_;

            bool destroyed_;
            std::chrono::milliseconds requestTimeout_;
            quasar::NamedCallbackQueue timerQueue_{"Transport"};

            SocketApi::DataSubscription dataSubscription_;
            SocketApi::StatusSubscription statusSubscription_;

            ObserverList<const std::string& /*payload*/> pushObservers_;
        };

    } // namespace xiva
} // namespace messenger
