#include "alice_capability_proxy.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

AliceCapabilityProxy::AliceCapabilityProxy(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                           std::shared_ptr<quasar::ICallbackQueue> worker)
    : IRemoteObject(std::move(remotingRegistry))
    , callbackQueue_(std::move(worker))
    , remoteObjectId_("AliceCapability")
{
}

AliceCapabilityProxy::~AliceCapabilityProxy()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(remoteObjectId_);
    }
}

void AliceCapabilityProxy::init()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(remoteObjectId_, weak_from_this());
    }
}

void AliceCapabilityProxy::handleRemotingConnect(std::shared_ptr<IRemotingConnection> connection) {
    callbackQueue_->add([this, connection]() {
        YIO_LOG_INFO("Connected to " << remoteObjectId_);
        connection_ = connection;
    });
}

void AliceCapabilityProxy::startRequest(
    std::shared_ptr<VinsRequest> request,
    std::shared_ptr<IAliceRequestEvents> events)
{
    callbackQueue_->add([this, request = std::move(request), events = std::move(events)]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_alice_capability_method();
        method->set_method(quasar::proto::Remoting::AliceCapabilityMethod::START_REQUEST);
        method->mutable_vins_request()->CopyFrom(VinsRequest::convertToVinsRequestProtobuf(request));
        method->set_events(events != nullptr);

        if (events != nullptr) {
            requestsById_.emplace(request->getId(),
                                  Request{
                                      .request = request,
                                      .events = events});
        }

        trySendMessage(remoting);
    });
}

void AliceCapabilityProxy::cancelDialog()
{
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_alice_capability_method();
        method->set_method(quasar::proto::Remoting::AliceCapabilityMethod::CANCEL_DIALOG);

        trySendMessage(remoting);
    });
}

void AliceCapabilityProxy::cancelDialogAndClearQueue()
{
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_alice_capability_method();
        method->set_method(quasar::proto::Remoting::AliceCapabilityMethod::CANCEL_DIALOG_AND_CLEAR_QUEUE);

        trySendMessage(remoting);
    });
}

void AliceCapabilityProxy::startConversation(const VinsRequest::EventSource& eventSource)
{
    callbackQueue_->add([this, eventSource]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_alice_capability_method();
        method->set_method(quasar::proto::Remoting::AliceCapabilityMethod::START_CONVERSATION);
        method->mutable_vins_request()->mutable_event_source()->CopyFrom(eventSource);

        trySendMessage(remoting);
    });
}

void AliceCapabilityProxy::stopConversation()
{
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_alice_capability_method();
        method->set_method(quasar::proto::Remoting::AliceCapabilityMethod::STOP_CONVERSATION);

        trySendMessage(remoting);
    });
}

void AliceCapabilityProxy::toggleConversation(const VinsRequest::EventSource& eventSource)
{
    callbackQueue_->add([this, eventSource]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_alice_capability_method();
        method->set_method(quasar::proto::Remoting::AliceCapabilityMethod::TOGGLE_CONVERSATION);
        method->mutable_vins_request()->mutable_event_source()->CopyFrom(eventSource);

        trySendMessage(remoting);
    });
}

void AliceCapabilityProxy::finishConversationVoiceInput()
{
    callbackQueue_->add([this]() {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId_));

        auto method = remoting.mutable_alice_capability_method();
        method->set_method(quasar::proto::Remoting::AliceCapabilityMethod::FINISH_CONVERSATION_VOICE_INPUT);

        trySendMessage(remoting);
    });
}

void AliceCapabilityProxy::addListener(std::weak_ptr<IAliceCapabilityListener> listener)
{
    listeners_.insert(std::move(listener));
}

void AliceCapabilityProxy::removeListener(std::weak_ptr<IAliceCapabilityListener> listener)
{
    listeners_.erase(listener);
}

void AliceCapabilityProxy::handleRemotingMessage(const quasar::proto::Remoting& message,
                                                 std::shared_ptr<IRemotingConnection> /*connection*/) {
    callbackQueue_->add([this, message]() {
        handleRemotingMessageImpl(message);
    });
}

void AliceCapabilityProxy::handleRemotingMessageImpl(const quasar::proto::Remoting& message)
{
    if (message.has_alice_request_events_method()) {
        const auto& method = message.alice_request_events_method();

        auto iter = requestsById_.find(method.request_id());
        if (iter == requestsById_.end()) {
            return;
        }
        auto request = iter->second.request;
        auto events = iter->second.events;

        switch (method.method()) {
            case quasar::proto::Remoting::AliceRequestEventsMethod::ALICE_REQUEST_STARTED: {
                events->onAliceRequestStarted(std::move(request));
                break;
            }
            case quasar::proto::Remoting::AliceRequestEventsMethod::ALICE_REQUEST_COMPLETED: {
                requestsById_.erase(iter);
                try {
                    auto json = quasar::parseJson(method.vins_response_json());
                    events->onAliceRequestCompleted(std::move(request), json);
                } catch (const std::exception& e) {
                    events->onAliceRequestError(std::move(request), "failedToParseJson", e.what());
                }

                break;
            }
            case quasar::proto::Remoting::AliceRequestEventsMethod::ALICE_REQUEST_ERROR: {
                requestsById_.erase(iter);
                events->onAliceRequestError(std::move(request), method.error_code(), method.error_text());
                break;
            }
        }
    }
    if (message.has_alice_capability_listener_method()) {
        const auto& method = message.alice_capability_listener_method();

        switch (method.method()) {
            case quasar::proto::Remoting::AliceCapabilityListenerMethod::STATE_CHANGED: {
                for (const auto& listener : listeners_) {
                    if (const auto& slistener = listener.lock()) {
                        slistener->onAliceStateChanged(method.state());
                    }
                }
                break;
            }
            case quasar::proto::Remoting::AliceCapabilityListenerMethod::TTS_COMPLETED: {
                for (const auto& listener : listeners_) {
                    if (const auto& slistener = listener.lock()) {
                        slistener->onAliceTtsCompleted();
                    }
                }
                break;
            }
        }
    }
}

void AliceCapabilityProxy::trySendMessage(const quasar::proto::Remoting& message) {
    if (!connection_) {
        YIO_LOG_WARN("Not connected to " << remoteObjectId_);
        return;
    }
    connection_->sendMessage(message);
}
